# -*- coding: utf-8 -*-
"""
ELQE Analyser 2.0

For rapid analysis of files produced from large quantities of ELQE measurements in the laboratories of Henry James Snaith (Department of Physics, University of Oxford). Specifically for the handling of .ilv, .ilv_processed, .tiv, .tiv_processed, and .spa, generated by LabView programs from the G44 Glovebox ELQE setup and the G46 integrating sphere setup, with or without post-processing (_processed) with scripts originally developed by Jonathan Warby (GloveboxLEDanalysis121119.py and GloveboxLEDanalysis081119time.py, and derivatives thereof).

Developed by Emil Grove Dyrvik (Department of Physics, University of Oxford)
Development started 21/02/2020
Version 1.0 and 1.1 completed 26/02/2020
Version 2.0 completed 27/01/2021

Contact:
emil.dyrvik@physics.ox.ac.uk
emil.dyrvik@gmail.com
"""

import numpy as np
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib as mpl
import seaborn as sns
import pandas as pd
import os
import datetime
import pathlib


""""Get the time"""
time_start = datetime.datetime.now()
date = datetime.date.today()

"""Global colourmaps for plots"""
CM_SUBSECTION_GREENS = np.linspace(0.4, 1.0, 1000) 
COLOURS_GREENS = [ cm.Greens(x) for x in CM_SUBSECTION_GREENS ]

CM_SUBSECTION_SPECTRAL = np.linspace(0, 1.0, 1000) 
COLOURS_SPECTRAL = [ cm.nipy_spectral(x) for x in CM_SUBSECTION_SPECTRAL ]

CM_SUBSECTION_ORANGES = np.linspace(0.3, 1.0, 1000) 
COLOURS_ORANGES = [ cm.Oranges(x) for x in CM_SUBSECTION_ORANGES ]

"""Global constant defaults (prompted for user definition at start)"""
MEASUREMENT_TYPE_DEFAULT = 1 # scans = 1, timed = 2
FILE_TYPE_SWITCH_DEFAULT = 1 # G44 unprocessed = 1, G44 processed = 2, G46 integrating sphere = 3
ARCHITECTURE_TYPE_DEFAULT = 1 # p-i-n = 1, n-i-p = 2

DATA_FOLDER_NAME_DEFAULT = 'Raw data' # must be a subfolder of the script
EXPORT_FOLDER_NAME_DEFAULT_BASE = 'Analysis ' + date.strftime('%Y-%m-%d')

VOLTAGE_THRESHOLD_DEFAULT = 3
REALISTIC_EQE_THRESHOLD_DEFAULT = 25
RADIANCE_THRESHOLD_DEFAULT = 5e-3
NOISE_THRESHOLD_DEFAULT = 0.75

"""Prompt for measurement type, file origin, and analysis parameter values"""
def prompt_user_input():
    """"Prompt if interested in analysing scans or timed measurements"""
    temporary_input = input('Select measurement type for analysis (scans = 1, timed = 2): ')
    if int(temporary_input) == 99: # shortcut to default everything
        return [MEASUREMENT_TYPE_DEFAULT,\
                FILE_TYPE_SWITCH_DEFAULT,\
                ARCHITECTURE_TYPE_DEFAULT,\
                DATA_FOLDER_NAME_DEFAULT,\
                EXPORT_FOLDER_NAME_DEFAULT_BASE,\
                VOLTAGE_THRESHOLD_DEFAULT,\
                REALISTIC_EQE_THRESHOLD_DEFAULT,\
                RADIANCE_THRESHOLD_DEFAULT,\
                NOISE_THRESHOLD_DEFAULT]
    elif int(temporary_input) == 1 or int(temporary_input) == 2:
        measurement_type = int(temporary_input)
    else:
        measurement_type = MEASUREMENT_TYPE_DEFAULT
    
    """"Setting function-internal booleans for the different measurement types (scan/timed)"""
    if measurement_type == 1:
        type_scan_internal = True
    elif measurement_type == 2:
        type_scan_internal = False
    type_timed_internal = not type_scan_internal
    
    """Ask what type of files to search for"""    
    temporary_input = input('Select file origin (G44 unprocessed = 1, G44 processed = 2, G46 integrating sphere = 3): ')
    if int(temporary_input) == 1 or int(temporary_input) == 2 or int(temporary_input) == 3:
        file_type_switch = int(temporary_input)
    else:
        file_type_switch = FILE_TYPE_SWITCH_DEFAULT
    
    """Ask what the architecture of the device was"""    
    temporary_input = input('Select device architecture (p-i-n = 1, n-i-p = 2): ')
    if int(temporary_input) == 1 or int(temporary_input) == 2:
        architecture_type = int(temporary_input)
    else:
        architecture_type = ARCHITECTURE_TYPE_DEFAULT    
    
    """Ask where the data is stored"""
    temporary_string = 'Enter name of raw data folder (must be a subfolder of the location of the script), or enter 0 for default (' + DATA_FOLDER_NAME_DEFAULT + '): '
    temporary_input = input(temporary_string)
    if len(temporary_input) == 1:
        if int(temporary_input) == 0:
            data_folder_name = DATA_FOLDER_NAME_DEFAULT
    else:
        data_folder_name = temporary_input
    
    """Ask where the analysis should be exported"""
    if type_scan_internal:
        export_folder_name_addon_type = 'scans'
    elif type_timed_internal:
        export_folder_name_addon_type = 'timed'
        
    if file_type_switch == 1:
        export_folder_name_addon_origin = 'G44 unprocessed'
    if file_type_switch == 2:
        export_folder_name_addon_origin = 'G44 processed'
    if file_type_switch == 3:
        export_folder_name_addon_origin = 'G46 integrating sphere'        
    
    export_folder_name_default = EXPORT_FOLDER_NAME_DEFAULT_BASE + ' (' + export_folder_name_addon_type + ', ' + export_folder_name_addon_origin + ')' 
    temporary_string = 'Determine a name for the export folder, or enter 0 for default (' + export_folder_name_default + '): '
    temporary_input = input(temporary_string)
    if len(temporary_input) == 1:
        if int(temporary_input) == 0:
            export_folder_name = export_folder_name_default
    else:
        export_folder_name = temporary_input
    
    """Voltage threshold value (only for scans)"""
    if type_scan_internal:
        temporary_string = 'Set minimum voltage threshold for the automated identification of peak EQE values, or enter 0 for default (' + str(VOLTAGE_THRESHOLD_DEFAULT) + 'V). Enter without units: '
        temporary_input = input(temporary_string)
        if int(temporary_input) == 0:
            voltage_threshold = VOLTAGE_THRESHOLD_DEFAULT
        else:
            voltage_threshold = float(temporary_input)
    else:
        voltage_threshold = VOLTAGE_THRESHOLD_DEFAULT

    """Realistic EQE threshold value (for both scans and timed)"""
    temporary_string = 'Set maximum realistic EQE threshold for the automated identification of peak EQE values in scan type measurements or maximum EQE values in timed type measurements, or enter 0 for default (' + str(REALISTIC_EQE_THRESHOLD_DEFAULT) + '%). Enter without units: '
    temporary_input = input(temporary_string)
    if int(temporary_input) == 0:
        realistic_eqe_threshold = REALISTIC_EQE_THRESHOLD_DEFAULT
    else:
        realistic_eqe_threshold = float(temporary_input)

    """Radiance threshold value (only for scans)"""
    if type_scan_internal:
        temporary_string = 'Set radiance threshold for identification of peak EQE, peak radiance, and luminescence turn-on, or enter 0 for default (' + str(RADIANCE_THRESHOLD_DEFAULT) + f' W sr\N{SUPERSCRIPT MINUS}\N{SUPERSCRIPT ONE} m\N{SUPERSCRIPT MINUS}\N{SUPERSCRIPT TWO}' + '). Enter without units: ' # using named unicode letters for the superscripts
        temporary_input = input(temporary_string)
        if int(temporary_input) == 0:
            radiance_threshold = RADIANCE_THRESHOLD_DEFAULT
        else:
            radiance_threshold = float(temporary_input)  
    else:
        radiance_threshold = RADIANCE_THRESHOLD_DEFAULT
    
    """Noise threshold value (only for timed). Strictly this isn't a threshold but a factor."""
    if type_timed_internal:
        temporary_string = 'Set noise threshold for the automated identification of maximum EQE values in timed measurements, or enter 0 for default (' + str(NOISE_THRESHOLD_DEFAULT) + '). Enter without units: '
        temporary_input = input(temporary_string)
        if int(temporary_input) == 0:
            noise_threshold = NOISE_THRESHOLD_DEFAULT
        else:
            noise_threshold = float(temporary_input)
    else:
        noise_threshold = NOISE_THRESHOLD_DEFAULT
        
    return [measurement_type,\
            file_type_switch,\
            architecture_type,\
            data_folder_name,\
            export_folder_name,\
            voltage_threshold,\
            realistic_eqe_threshold,\
            radiance_threshold,\
            noise_threshold]
    
"""Write one or multiple strings to the log file"""
def write_to_log(to_log):
    """to_log is either a string or a list of strings, without newline character \n"""
    log_file = open(log_file_path,'a')
    if isinstance(to_log,str):
        string = to_log + '\n'
        log_file.write(string)
    else:
        for i in range(len(to_log)):
            string = to_log[i] + '\n'
            log_file.write(string)
            
    log_file.close()

"""Finding scan type files for G44 unprocessed"""
def file_finder_ilv():
    table_of_measurement_files_list = []
    
    print('Finding .ilv files in', data_folder_path)
    for filename in os.listdir(data_folder_path): # finding relevant files
        if filename.endswith(".ilv"):
            table_of_measurement_files_list.append(fetch_scan_info_unprocessed(filename))
                        
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (scan)']) # creating dataframe for file info
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
           
    return [table_of_measurement_files_df, file_numbers_list]

"""Finding scan type files for G44 processed"""
def file_finder_ilv_processed():
    table_of_measurement_files_list = []
    
    print('Finding .ilv_processed files in', data_folder_path)
    for filename in os.listdir(data_folder_path): # finding relevant files
        if filename.endswith(".ilv_processed"):
            table_of_measurement_files_list.append(fetch_scan_info_processed(filename))
                        
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (scan)']) # creating dataframe for file info
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
        
    return [table_of_measurement_files_df, file_numbers_list]

"""Finding scan type files for G46 integrating sphere"""
def file_finder_ilv_spa():
    table_of_measurement_files_list = []
    
    print('Finding .ilv files with matching .spa files in', data_folder_path)
    for filename_ilv in os.listdir(data_folder_path):
        if filename_ilv.endswith('.ilv'):
            for filename_spa in os.listdir(data_folder_path):
                if filename_spa.endswith('.spa') and filename_spa[0:-4] == filename_ilv[0:-4]:
                    table_of_measurement_files_list.append(fetch_scan_info_unprocessed(filename_spa))
                        
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (spectrum)', 'File name (scan)']) # creating dataframe for file info
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
           
    return [table_of_measurement_files_df, file_numbers_list]

"""Finding timed type files for G44 unprocessed"""
def file_finder_tiv():
    table_of_measurement_files_list = []
    
    print('Finding .tiv files in', data_folder_path)
    for filename in os.listdir(data_folder_path):
        if filename.endswith(".tiv"):
            table_of_measurement_files_list.append(fetch_scan_info_unprocessed(filename))
            
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (timed)'])
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))    
           
    return [table_of_measurement_files_df, file_numbers_list]

"""Finding timed type files for G44 processed"""
def file_finder_tiv_processed():
    table_of_measurement_files_list = []
    
    print('Finding .tiv_processed files in', data_folder_path)
    for filename in os.listdir(data_folder_path):
        if filename.endswith(".tiv_processed"):
            table_of_measurement_files_list.append(fetch_scan_info_processed(filename))
            
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (timed)'])
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))    
           
    return [table_of_measurement_files_df, file_numbers_list]

"""Finding timed type files for G46 integrating sphere"""
def file_finder_tiv_spa():
    table_of_measurement_files_list = []
    
    print('Finding .tiv files with matching .spa files in', data_folder_path)
    for filename_tiv in os.listdir(data_folder_path):
        if filename_tiv.endswith('.tiv'):
            for filename_spa in os.listdir(data_folder_path):
                if filename_spa.endswith('.spa') and filename_spa[0:-4] == filename_tiv[0:-4]:
                    table_of_measurement_files_list.append(fetch_scan_info_unprocessed(filename_spa))
                        
    table_of_measurement_files_df = pd.DataFrame(table_of_measurement_files_list, columns=['Experiment name','Substrate label','Pixel','Scan #','File name (spectrum)', 'File name (timed)'])
    table_of_measurement_files_df['Experiment name'] = table_of_measurement_files_df['Experiment name'].str.lower() # removing case sensitivity
    table_of_measurement_files_df['Substrate label'] = table_of_measurement_files_df['Substrate label'].str.lower() # removing case sensitivity
    
    file_numbers_list = fetch_file_numbers(table_of_measurement_files_df)
    print('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
    write_to_log('Found %i experiment(s), %i substrate(s), %i pixel(s), and %i scan(s).' % (file_numbers_list[0], file_numbers_list[1], file_numbers_list[2], file_numbers_list[3]))
           
    return [table_of_measurement_files_df, file_numbers_list]

"""Analysing the filename and suffix to fetch information about the measurement"""
"""For G44 unprocessed files and files from G46 integrating sphere"""
def fetch_scan_info_unprocessed(filename):
    split_underscore = filename.split('_')
    split_dot = split_underscore[-1].split('.')
    experiment_name = split_underscore[-4]
    substrate_label = split_underscore[-3]
    pixel_number = int(split_underscore[-2])
    scan_number = int(split_dot[0])
    if g46_integrating_sphere and type_scan:
        filename_ilv = filename[0:-3] + 'ilv' # changing from .spa to .ilv (to save both)
        return [experiment_name, substrate_label, pixel_number, scan_number, filename, filename_ilv]
    elif g46_integrating_sphere and type_timed:
        filename_ilv = filename[0:-3] + 'tiv' # changing from .spa to .ilv (to save both)
        return [experiment_name, substrate_label, pixel_number, scan_number, filename, filename_ilv]    
    else:
        return [experiment_name, substrate_label, pixel_number, scan_number, filename]

"""Analysing the filename and suffix to fetch information about the measurement"""
"""For G44 processed files"""
def fetch_scan_info_processed(filename):
    split_underscore = filename.split('_')
    split_dot = split_underscore[-2].split('.')
    experiment_name = split_underscore[-5]
    substrate_label = split_underscore[-4]
    pixel_number = int(split_underscore[-3])
    scan_number = int(split_dot[0])
    return [experiment_name, substrate_label, pixel_number, scan_number, filename]

"""Finding the number of different experiments, substrates, pixels, and scans for a set of datafiles"""
def fetch_file_numbers(table_of_measurement_files_df): 
    """table_of_measurement_files_df should already be case insensitivised """
    number_of_experiments = table_of_measurement_files_df.groupby(['Experiment name']).ngroups
    number_of_substrates = table_of_measurement_files_df.groupby(['Experiment name','Substrate label']).ngroups
    number_of_pixels = table_of_measurement_files_df.groupby(['Experiment name','Substrate label','Pixel']).ngroups
    number_of_scans = table_of_measurement_files_df.groupby(['Experiment name','Substrate label','Pixel','Scan #']).ngroups
    return [number_of_experiments, number_of_substrates, number_of_pixels, number_of_scans]

"""Extracts the modification time of a measurement file to determine when the measurement was made or processed"""
def fetch_file_creation_time(filename):
    file_path = os.path.join(data_folder_path,filename)
    file_path_pathlib = pathlib.Path(file_path)
    file_creation_time = datetime.datetime.fromtimestamp(file_path_pathlib.stat().st_mtime)
    
    return file_creation_time
    
"""Finding the peak EQE of a set of scans and associated values"""
def analyse_peak_eqe(series_of_data_file_names):
    """Pre-allocating"""
    list_of_max_eqe = []
    list_of_voltage_at_max_eqe = []
    list_of_current_at_max_eqe = []
    list_of_radiance_at_max_eqe = []
    array_size = len(series_of_data_file_names)
    for scan in range(array_size):
        """Loading data file"""
        file_path = os.path.join(data_folder_path,series_of_data_file_names.iloc[scan])
        if g44_unprocessed or g46_integrating_sphere:
            scan_data_df = pd.read_csv(file_path, delimiter='\t')
        elif g44_processed:
            scan_data_df = pd.read_csv(file_path)
            
        """Flip the sign of current density and voltage data if necessary"""
        if device_nip and not g46_integrating_sphere:
            scan_data_df['LED voltage [V]'] = -scan_data_df['LED voltage [V]']
            scan_data_df['LED current [mA/cm2]'] = -scan_data_df['LED current [mA/cm2]']
        elif device_pin and g46_integrating_sphere:
            scan_data_df['Voltage [V]'] = -scan_data_df['Voltage [V]']
            scan_data_df['Current [mA/cm2]'] = -scan_data_df['Current [mA/cm2]']
            
        """Return NaNs if there is so few values in the scan that there is no point in analysing"""
        if len(scan_data_df) < 2:
            list_of_max_eqe.append(np.nan)
            list_of_voltage_at_max_eqe.append(np.nan)
            list_of_current_at_max_eqe.append(np.nan)
            list_of_radiance_at_max_eqe.append(np.nan)
            continue
        
        """Truncating to ignore values below the voltage threshold"""
        if g44_unprocessed or g44_processed:
            truncated_series_voltage = scan_data_df['LED voltage [V]'].ge(voltage_threshold)
        elif g46_integrating_sphere:
            truncated_series_voltage = scan_data_df['Voltage [V]'].ge(voltage_threshold)
        
        scan_data_df_truncated_voltage = scan_data_df.truncate(before=len(scan_data_df)-sum(truncated_series_voltage))
        
        """Truncating further to ignore values below the radiance threshold"""
        truncated_series_radiance = scan_data_df_truncated_voltage['Radiance [W/sr/m2]'].ge(radiance_threshold)
        scan_data_df_truncated = scan_data_df_truncated_voltage.truncate(before=scan_data_df_truncated_voltage.index[-1]+1-sum(truncated_series_radiance))
        
        """Finding the max EQE, checking if it's realistic and cleaning if necessary"""
        if len(scan_data_df_truncated) > 0:
            max_eqe_index = scan_data_df_truncated['EQE [%]'].idxmax()
            if scan_data_df_truncated['EQE [%]'][max_eqe_index] > realistic_eqe_threshold:
                while scan_data_df_truncated['EQE [%]'][max_eqe_index] > realistic_eqe_threshold:
                    filename = series_of_data_file_names.iloc[scan]
                    if g44_unprocessed or g46_integrating_sphere:
                        scan_info = fetch_scan_info_unprocessed(filename)
                    elif g44_processed:
                        scan_info = fetch_scan_info_processed(filename)                
       
                    print('(Peak EQE Search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value)' % (scan_info[3], scan_info[1],  scan_info[2]))
                    write_to_log(['(Peak EQE Search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value.)' % (scan_info[3], scan_info[1],  scan_info[2]), 'The value was %f' % scan_data_df_truncated['EQE [%]'][max_eqe_index]])           
                    
                    """Remove the value and find new max, but only if there's still more data points left in the dataframe"""
                    scan_data_df_truncated = scan_data_df_truncated.drop([max_eqe_index])
                    if len(scan_data_df_truncated) > 0:
                        max_eqe_index = scan_data_df_truncated['EQE [%]'].idxmax()
                    else:
                        print('No useful values in this scan: substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                        write_to_log('No useful values in this scan: substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                        break
                            
            """Append max of this scan to list of max EQEs (but only if we're sure that the scan has at least 1 useful data point, otherwise append NaNs)"""
            if len(scan_data_df_truncated) > 0:
                if g44_processed or g44_unprocessed:
                    list_of_max_eqe.append(scan_data_df_truncated['EQE [%]'][max_eqe_index])
                    list_of_voltage_at_max_eqe.append(scan_data_df_truncated['LED voltage [V]'][max_eqe_index])
                    list_of_current_at_max_eqe.append(scan_data_df_truncated['LED current [mA/cm2]'][max_eqe_index])
                    list_of_radiance_at_max_eqe.append(scan_data_df_truncated['Radiance [W/sr/m2]'][max_eqe_index])
                elif g46_integrating_sphere:
                    list_of_max_eqe.append(scan_data_df_truncated['EQE [%]'][max_eqe_index])
                    list_of_voltage_at_max_eqe.append(scan_data_df_truncated['Voltage [V]'][max_eqe_index])
                    list_of_current_at_max_eqe.append(scan_data_df_truncated['Current [mA/cm2]'][max_eqe_index])
                    list_of_radiance_at_max_eqe.append(scan_data_df_truncated['Radiance [W/sr/m2]'][max_eqe_index])
            else:
                list_of_max_eqe.append(np.nan)
                list_of_voltage_at_max_eqe.append(np.nan)
                list_of_current_at_max_eqe.append(np.nan)
                list_of_radiance_at_max_eqe.append(np.nan)
        else:
            list_of_max_eqe.append(np.nan)
            list_of_voltage_at_max_eqe.append(np.nan)
            list_of_current_at_max_eqe.append(np.nan)
            list_of_radiance_at_max_eqe.append(np.nan)
            
    """Select the peak EQE and find associated values, return all as NaN if there is no peak EQE detected"""
    if not len(list_of_max_eqe) == array_size:
        print('Error in analyis_peak_EQE: number of input scans does not match number of output analysis values.') # To check that I've done things correctly here
        # if this doesn't match up then the paragraph below is wrong too (I believe)
    
    if np.isnan(list_of_max_eqe).all(): # all elements of list are nans
        peak_eqe = np.nan
        voltage_at_peak_eqe = np.nan
        current_at_peak_eqe = np.nan
        radiance_at_peak_eqe = np.nan
        scan_number_of_peak_eqe = np.nan
        list_of_luminescence_turn_on_values_for_peak_eqe_scan = [np.nan, np.nan, np.nan, np.nan]
    else:
        peak_eqe = np.nanmax(list_of_max_eqe) 
        peak_eqe_index = np.nanargmax(list_of_max_eqe)
        voltage_at_peak_eqe = list_of_voltage_at_max_eqe[peak_eqe_index]
        current_at_peak_eqe = list_of_current_at_max_eqe[peak_eqe_index]
        radiance_at_peak_eqe = list_of_radiance_at_max_eqe[peak_eqe_index]
        filename = series_of_data_file_names.iloc[peak_eqe_index]
        if g44_unprocessed or g46_integrating_sphere:
            scan_info = fetch_scan_info_unprocessed(filename)
        elif g44_processed:
            scan_info = fetch_scan_info_processed(filename)     
        scan_number_of_peak_eqe = scan_info[3]
        list_of_luminescence_turn_on_values_for_peak_eqe_scan = analyse_luminescence_turn_on(series_of_data_file_names.iloc[peak_eqe_index])
    
    return [peak_eqe, voltage_at_peak_eqe, current_at_peak_eqe, radiance_at_peak_eqe, scan_number_of_peak_eqe, list_of_luminescence_turn_on_values_for_peak_eqe_scan]

"""Finding the peak radiance of a set of scans and associated values"""
def analyse_peak_radiance(series_of_data_file_names):
    """Pre-allocating"""
    list_of_max_radiance = []
    list_of_voltage_at_max_radiance = []
    list_of_current_at_max_radiance = []
    list_of_eqe_at_max_radiance = []
    array_size = len(series_of_data_file_names)   
    for scan in range(array_size):
        """Loading data file"""
        file_path = os.path.join(data_folder_path,series_of_data_file_names.iloc[scan])
        if g44_unprocessed or g46_integrating_sphere:
            scan_data_df = pd.read_csv(file_path, delimiter='\t')
        elif g44_processed:
            scan_data_df = pd.read_csv(file_path)
        
        """Flip the sign of current density and voltage data if necessary"""
        if device_nip and not g46_integrating_sphere:
            scan_data_df['LED voltage [V]'] = -scan_data_df['LED voltage [V]']
            scan_data_df['LED current [mA/cm2]'] = -scan_data_df['LED current [mA/cm2]']
        elif device_pin and g46_integrating_sphere:
            scan_data_df['Voltage [V]'] = -scan_data_df['Voltage [V]']
            scan_data_df['Current [mA/cm2]'] = -scan_data_df['Current [mA/cm2]']
        
        """Return NaNs if there is so few values in the scan that there is no point in analysing"""
        if len(scan_data_df) < 2:
            list_of_max_radiance.append(np.nan)
            list_of_voltage_at_max_radiance.append(np.nan)
            list_of_current_at_max_radiance.append(np.nan)
            list_of_eqe_at_max_radiance.append(np.nan)
            continue        
        
        """Truncating to ignore values below the radiance threshold"""
        truncated_series_radiance = scan_data_df['Radiance [W/sr/m2]'].ge(radiance_threshold)
        scan_data_df_truncated = scan_data_df.truncate(before=len(scan_data_df)-sum(truncated_series_radiance))        
        
        """Finding the max radiance of the scan, and append to the list. Append with NaNs if there are no values above threshold."""
        if len(scan_data_df_truncated) > 0:        
            if g44_unprocessed or g44_processed:
                max_radiance_index = scan_data_df_truncated['Radiance [W/sr/m2]'].idxmax()
                list_of_max_radiance.append(scan_data_df_truncated['Radiance [W/sr/m2]'][max_radiance_index])
                list_of_voltage_at_max_radiance.append(scan_data_df_truncated['LED voltage [V]'][max_radiance_index])
                list_of_current_at_max_radiance.append(scan_data_df_truncated['LED current [mA/cm2]'][max_radiance_index])
                list_of_eqe_at_max_radiance.append(scan_data_df_truncated['EQE [%]'][max_radiance_index])
            elif g46_integrating_sphere:
                max_radiance_index = scan_data_df_truncated['Radiance [W/sr/m2]'].idxmax()
                list_of_max_radiance.append(scan_data_df_truncated['Radiance [W/sr/m2]'][max_radiance_index])
                list_of_voltage_at_max_radiance.append(scan_data_df_truncated['Voltage [V]'][max_radiance_index])
                list_of_current_at_max_radiance.append(scan_data_df_truncated['Current [mA/cm2]'][max_radiance_index])
                list_of_eqe_at_max_radiance.append(scan_data_df_truncated['EQE [%]'][max_radiance_index])
        else:
            list_of_max_radiance.append(np.nan)
            list_of_voltage_at_max_radiance.append(np.nan)
            list_of_current_at_max_radiance.append(np.nan)
            list_of_eqe_at_max_radiance.append(np.nan)
            
    """Select the peak radiance and find associated values"""
    if not len(list_of_max_radiance) == array_size:
        print('Error in analyis_peak_radiance: number of input scans does not match number of output analysis values.') # To check that I've done things correctly here
        # if this doesn't match up then the paragraph below is wrong too (I believe)    
    
    if np.isnan(list_of_max_radiance).all(): # all elements of list are nans
        peak_radiance = np.nan
        voltage_at_peak_radiance = np.nan
        current_at_peak_radiance = np.nan
        eqe_at_peak_radiance = np.nan
        scan_number_of_peak_radiance = np.nan
        list_of_luminescence_turn_on_values_for_peak_radiance_scan = [np.nan, np.nan, np.nan, np.nan]
    else:
        peak_radiance = np.nanmax(list_of_max_radiance) 
        peak_radiance_index = np.nanargmax(list_of_max_radiance)
        voltage_at_peak_radiance = list_of_voltage_at_max_radiance[peak_radiance_index]
        current_at_peak_radiance = list_of_current_at_max_radiance[peak_radiance_index]
        eqe_at_peak_radiance = list_of_eqe_at_max_radiance[peak_radiance_index]
        filename = series_of_data_file_names.iloc[peak_radiance_index]
        if g44_unprocessed or g46_integrating_sphere:
            scan_info = fetch_scan_info_unprocessed(filename)
        elif g44_processed:
            scan_info = fetch_scan_info_processed(filename)     
        scan_number_of_peak_radiance = scan_info[3]
        list_of_luminescence_turn_on_values_for_peak_radiance_scan = analyse_luminescence_turn_on(series_of_data_file_names.iloc[peak_radiance_index])
    
    return [peak_radiance, voltage_at_peak_radiance, current_at_peak_radiance, eqe_at_peak_radiance, scan_number_of_peak_radiance, list_of_luminescence_turn_on_values_for_peak_radiance_scan]

"""Finding the luminescence turn-on for a single scan. Defined by radiance above a threshold"""
def analyse_luminescence_turn_on(filename):
    file_path = os.path.join(data_folder_path,filename)
    if g44_unprocessed or g46_integrating_sphere:
        scan_data_df = pd.read_csv(file_path, delimiter='\t')
    elif g44_processed:
        scan_data_df = pd.read_csv(file_path)
    
    """Flip the sign of current density and voltage data if necessary"""
    if device_nip and not g46_integrating_sphere:
        scan_data_df['LED voltage [V]'] = -scan_data_df['LED voltage [V]']
        scan_data_df['LED current [mA/cm2]'] = -scan_data_df['LED current [mA/cm2]']
    elif device_pin and g46_integrating_sphere:
        scan_data_df['Voltage [V]'] = -scan_data_df['Voltage [V]']
        scan_data_df['Current [mA/cm2]'] = -scan_data_df['Current [mA/cm2]']
    
    """Truncating to ignore values below the radiance threshold"""
    truncated_series = scan_data_df['Radiance [W/sr/m2]'].ge(radiance_threshold)
    scan_data_df_truncated = scan_data_df.truncate(before=len(scan_data_df)-sum(truncated_series))
    
    """Finding the first data point above the radiance threshold"""
    if len(scan_data_df_truncated) > 0:
        if g44_unprocessed or g44_processed:
            radiance_at_threshold = scan_data_df_truncated['Radiance [W/sr/m2]'].iloc[0]
            voltage_at_radiance_threshold = scan_data_df_truncated['LED voltage [V]'].iloc[0]
            current_at_radiance_threshold = scan_data_df_truncated['LED current [mA/cm2]'].iloc[0]
            eqe_at_radiance_threshold = scan_data_df_truncated['EQE [%]'].iloc[0]
        elif g46_integrating_sphere:
            radiance_at_threshold = scan_data_df_truncated['Radiance [W/sr/m2]'].iloc[0]
            voltage_at_radiance_threshold = scan_data_df_truncated['Voltage [V]'].iloc[0]
            current_at_radiance_threshold = scan_data_df_truncated['Current [mA/cm2]'].iloc[0]
            eqe_at_radiance_threshold = scan_data_df_truncated['EQE [%]'].iloc[0]
    else:
        radiance_at_threshold = np.nan
        voltage_at_radiance_threshold = np.nan
        current_at_radiance_threshold = np.nan
        eqe_at_radiance_threshold = np.nan
    
    return radiance_at_threshold, voltage_at_radiance_threshold, current_at_radiance_threshold, eqe_at_radiance_threshold

"""Finding the max EQE for a single scan and associated values"""
def analyse_max_eqe(filename):
    file_path = os.path.join(data_folder_path,filename)
    if g44_unprocessed or g46_integrating_sphere:
        scan_data_df = pd.read_csv(file_path, delimiter='\t')
    elif g44_processed:
        scan_data_df = pd.read_csv(file_path)
            
    """Flip the sign of current density and voltage data if necessary"""
    if device_nip and not g46_integrating_sphere:
        scan_data_df['LED voltage [V]'] = -scan_data_df['LED voltage [V]']
        scan_data_df['LED current [mA/cm2]'] = -scan_data_df['LED current [mA/cm2]']
    elif device_pin and g46_integrating_sphere:
        scan_data_df['Voltage [V]'] = -scan_data_df['Voltage [V]']
        scan_data_df['Current [mA/cm2]'] = -scan_data_df['Current [mA/cm2]']
            
    """Find the max and check that it is realistic"""
    if len(scan_data_df) > 0:
        """Finding the max EQE, checking if it's realistic and cleaning if necessary"""
        max_eqe_index = scan_data_df['EQE [%]'].idxmax()
        if scan_data_df['EQE [%]'][max_eqe_index] > realistic_eqe_threshold:
            while scan_data_df['EQE [%]'][max_eqe_index] > realistic_eqe_threshold:
                if g44_unprocessed or g46_integrating_sphere:
                    scan_info = fetch_scan_info_unprocessed(filename)
                elif g44_processed:
                    scan_info = fetch_scan_info_processed(filename)                
   
                print('(Max EQE Search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value)' % (scan_info[3], scan_info[1],  scan_info[2]))
                write_to_log(['(Max EQE Search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value.)' % (scan_info[3], scan_info[1],  scan_info[2]), 'The value was %f' % scan_data_df['EQE [%]'][max_eqe_index]])           
                
                """Remove the value and find new max, but only if there's still more data points left in the dataframe"""
                scan_data_df = scan_data_df.drop([max_eqe_index])
                if len(scan_data_df) > 0:
                    max_eqe_index = scan_data_df['EQE [%]'].idxmax()
                else:
                    print('No useful values in this scan: substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                    write_to_log('No useful values in this scan: substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                    break
                
    """Find values related to max EQE, or return if no data points left to evaluate"""
    if len(scan_data_df) > 0:
        if g44_processed or g44_unprocessed:
            radiance_at_max_eqe = scan_data_df['Radiance [W/sr/m2]'][max_eqe_index]
            voltage_at_max_eqe = scan_data_df['LED voltage [V]'][max_eqe_index]
            current_at_max_eqe = scan_data_df['LED current [mA/cm2]'][max_eqe_index]
            eqe_at_max_eqe = scan_data_df['EQE [%]'][max_eqe_index]            
        elif g46_integrating_sphere:
            radiance_at_max_eqe = scan_data_df['Radiance [W/sr/m2]'][max_eqe_index]
            voltage_at_max_eqe = scan_data_df['Voltage [V]'][max_eqe_index]
            current_at_max_eqe = scan_data_df['Current [mA/cm2]'][max_eqe_index]
            eqe_at_max_eqe = scan_data_df['EQE [%]'][max_eqe_index]                   
    else:
        radiance_at_max_eqe = np.nan
        voltage_at_max_eqe = np.nan
        current_at_max_eqe = np.nan
        eqe_at_max_eqe = np.nan
        
    return radiance_at_max_eqe, voltage_at_max_eqe, current_at_max_eqe, eqe_at_max_eqe

"""Finding key values at a given voltage for a single scan"""
def analyse_voltage(filename):
    file_path = os.path.join(data_folder_path,filename)
    if g44_unprocessed or g46_integrating_sphere:
        scan_data_df = pd.read_csv(file_path, delimiter='\t')
    elif g44_processed:
        scan_data_df = pd.read_csv(file_path)
            
    """Flip the sign of current density and voltage data if necessary"""
    if device_nip and not g46_integrating_sphere:
        scan_data_df['LED voltage [V]'] = -scan_data_df['LED voltage [V]']
        scan_data_df['LED current [mA/cm2]'] = -scan_data_df['LED current [mA/cm2]']
    elif device_pin and g46_integrating_sphere:
        scan_data_df['Voltage [V]'] = -scan_data_df['Voltage [V]']
        scan_data_df['Current [mA/cm2]'] = -scan_data_df['Current [mA/cm2]']
    
    for V in [2, 2.5, 3, 3.5, 4]:
        if g44_unprocessed or g44_processed:
            """Find closest data point and the associated values. If there are no data points in this voltage range then return NaNs"""
            if scan_data_df['LED voltage [V]'][scan_data_df.last_valid_index()] > V-0.075:
                delta_series = scan_data_df['LED voltage [V]'] - V
                delta_series_absolutes = delta_series.abs()
                closest_data_point_index = delta_series_absolutes.idxmin()
                radiance = scan_data_df['Radiance [W/sr/m2]'][closest_data_point_index]
                voltage = scan_data_df['LED voltage [V]'][closest_data_point_index]
                current = scan_data_df['LED current [mA/cm2]'][closest_data_point_index]
                eqe = scan_data_df['EQE [%]'][closest_data_point_index]    
            else:
                radiance = np.nan
                voltage = np.nan
                current = np.nan
                eqe = np.nan
        elif g46_integrating_sphere:
            """Find closest data point and the associated values. If there are no data points in this voltage range then return NaNs"""
            if scan_data_df['Voltage [V]'][scan_data_df.last_valid_index()] > V-0.075:
                delta_series = scan_data_df['Voltage [V]'] - V
                delta_series_absolutes = delta_series.abs()
                closest_data_point_index = delta_series_absolutes.idxmin()
                radiance = scan_data_df['Radiance [W/sr/m2]'][closest_data_point_index]
                voltage = scan_data_df['Voltage [V]'][closest_data_point_index]
                current = scan_data_df['Current [mA/cm2]'][closest_data_point_index]
                eqe = scan_data_df['EQE [%]'][closest_data_point_index]    
            else:
                radiance = np.nan
                voltage = np.nan
                current = np.nan
                eqe = np.nan                
        
        """Assign value to return variables"""
        if V == 2:
            radiance_at_2_0_V = radiance
            voltage_at_2_0_V = voltage
            current_at_2_0_V = current
            eqe_at_2_0_V = eqe   
        elif V == 2.5:
            radiance_at_2_5_V = radiance
            voltage_at_2_5_V = voltage
            current_at_2_5_V = current
            eqe_at_2_5_V = eqe   
        elif V == 3:
            radiance_at_3_0_V = radiance
            voltage_at_3_0_V = voltage
            current_at_3_0_V = current
            eqe_at_3_0_V = eqe   
        elif V == 3.5:
            radiance_at_3_5_V = radiance
            voltage_at_3_5_V = voltage
            current_at_3_5_V = current
            eqe_at_3_5_V = eqe   
        elif V == 4:
            radiance_at_4_0_V = radiance
            voltage_at_4_0_V = voltage
            current_at_4_0_V = current
            eqe_at_4_0_V = eqe       
            
    return radiance_at_2_0_V, voltage_at_2_0_V, current_at_2_0_V, eqe_at_2_0_V,\
                radiance_at_2_5_V, voltage_at_2_5_V, current_at_2_5_V, eqe_at_2_5_V,\
                    radiance_at_3_0_V, voltage_at_3_0_V, current_at_3_0_V, eqe_at_3_0_V,\
                        radiance_at_3_5_V, voltage_at_3_5_V, current_at_3_5_V, eqe_at_3_5_V,\
                            radiance_at_4_0_V, voltage_at_4_0_V, current_at_4_0_V, eqe_at_4_0_V

"""Extracting the max EQE, time to max, and t50 and t80 values from initial and max"""
def analyse_timed_key_data(series_of_data_file_names):
    """Pre-allocating"""
    array_size = len(series_of_data_file_names)
    list_of_durations = [] # array with duration of each measurement
    list_of_max_eqe = [] # array with max EQE of each measurmeent
    list_of_time_to_max_eqe = [] # array with the time max EQE is reached for each measurement
    list_of_t50_from_max = [] # array with time from max EQE to t50 (50 % reduction from max EQE) 
    list_of_t50_from_initial = [] # array with time from first data point to t50 (50 % reduction from first data point)
    list_of_t80_from_max = []
    list_of_t80_from_initial = []
    """Loading files"""
    for scan in range(array_size):
        file_path = os.path.join(data_folder_path,series_of_data_file_names.iloc[scan])
        if g44_unprocessed or g46_integrating_sphere:
            timed_data_df = pd.read_csv(file_path, delimiter='\t')
        elif g44_processed:
            timed_data_df = pd.read_csv(file_path)        
        
        """Flip the sign of current density and voltage data if necessary"""
        # Not used in this version but included for potential future functionalities
        if device_nip and not g46_integrating_sphere:
            timed_data_df['LED voltage [V]'] = -timed_data_df['LED voltage [V]']
            timed_data_df['LED current [mA/cm2]'] = -timed_data_df['LED current [mA/cm2]']
        elif device_pin and g46_integrating_sphere:
            timed_data_df['Voltage [V]'] = -timed_data_df['Voltage [V]']
            timed_data_df['Current [mA/cm2]'] = -timed_data_df['Current [mA/cm2]']
        
        """Check that the data has enough data points that it makes sense to analyse, otherwise return NaNs"""
        if len(timed_data_df) < 2:
            list_of_durations.append(np.nan)
            list_of_max_eqe.append(np.nan)
            list_of_time_to_max_eqe.append(np.nan)
            list_of_t50_from_max.append(np.nan)
            list_of_t50_from_initial.append(np.nan)
            list_of_t80_from_max.append(np.nan)
            list_of_t80_from_initial.append(np.nan)
            continue
        
        """Find a candidate for max and check if it is realistic, clean if necessary"""
        """Value accepted if lower than a maximum realistic threshold value and not so much higher than the following value that it is likely to be a noise spike"""
        max_eqe_index = timed_data_df['EQE [%]'].idxmax()
        if timed_data_df['EQE [%]'][max_eqe_index+1] < timed_data_df['EQE [%]'][max_eqe_index]*noise_threshold or timed_data_df['EQE [%]'][max_eqe_index] > realistic_eqe_threshold: 
            while timed_data_df['EQE [%]'][max_eqe_index+1] < timed_data_df['EQE [%]'][max_eqe_index]*noise_threshold or timed_data_df['EQE [%]'][max_eqe_index] > realistic_eqe_threshold: 
                
                filename = series_of_data_file_names.iloc[scan]
                if g44_processed:
                    scan_info = fetch_scan_info_processed(filename)
                elif g44_unprocessed or g46_integrating_sphere:
                    scan_info = fetch_scan_info_unprocessed(filename)
                    
                print('(Timed key data search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value)' % (scan_info[3], scan_info[1],  scan_info[2]))
                write_to_log(['(Peak EQE Search) Cleaning: removing a single data point in scan # %i in substrate %s pixel %i. (Unrealistic max EQE value.)' % (scan_info[3], scan_info[1],  scan_info[2]), 'The value was %f' % timed_data_df['EQE [%]'][max_eqe_index]])        
                
                timed_data_df = timed_data_df.drop([max_eqe_index])
                if len(timed_data_df) > 1:
                    timed_data_df = timed_data_df.reset_index(drop=True)
                    max_eqe_index = timed_data_df['EQE [%]'].idxmax()
                else:
                    print('No useful values in this scan: substrate %s pixel %i scan %i' % (scan_info[1], scan_info[2], scan_info[3]))
                    write_to_log('No useful values in this scan: substrate %s pixel %i scan %i' % (scan_info[1], scan_info[2], scan_info[3]))                 
                    break
        
        """If still enough values remaining after cleaning, find all the key values, otherwise return NaNs."""                
        if len(timed_data_df) > 1:
            """If max is 0, then report all NaNs except for the duration"""
            if timed_data_df['EQE [%]'][max_eqe_index] == 0:
                list_of_durations.append(timed_data_df['Time[s]'][timed_data_df.last_valid_index()])
                list_of_max_eqe.append(np.nan)
                list_of_time_to_max_eqe.append(np.nan)
                list_of_t50_from_max.append(np.nan)
                list_of_t50_from_initial.append(np.nan)
                list_of_t80_from_max.append(np.nan)
                list_of_t80_from_initial.append(np.nan)  
            else:
                """Finding duration"""
                list_of_durations.append(timed_data_df['Time[s]'][timed_data_df.last_valid_index()])
                """Finding max and time to max"""
                list_of_max_eqe.append(timed_data_df['EQE [%]'][max_eqe_index])
                list_of_time_to_max_eqe.append(timed_data_df['Time[s]'][max_eqe_index])
                
                """Finding t50 from max"""
                """t50 from max has been changed from 1.1 to 1.2: In 1.1 it takes the time from start until a level that is 50% of max. In 1.2 it takes the time from reaching max to reaching 50% max."""
                timed_data_df_truncated_at_max = timed_data_df.truncate(before=max_eqe_index)
                boolean_series_t50 = timed_data_df_truncated_at_max['EQE [%]'].ge(0.5*timed_data_df['EQE [%]'][max_eqe_index]) # true for values above 50 %
                if sum(boolean_series_t50) < len(timed_data_df_truncated_at_max): # the sample has been degraded beyond t50 from max
                    t50_from_max_index = max_eqe_index + sum(boolean_series_t50)
                    list_of_t50_from_max.append(timed_data_df['Time[s]'][t50_from_max_index] - timed_data_df['Time[s]'][max_eqe_index])
                else:
                    list_of_t50_from_max.append(np.nan)
                    
                """Finding t50 from initial"""
                truncated_series = timed_data_df['EQE [%]'].le(0.5*timed_data_df['EQE [%]'][timed_data_df.first_valid_index()])
                timed_data_df_truncated = timed_data_df.truncate(before=len(timed_data_df)-sum(truncated_series)) # truncating to ignore values that are above 50% of initial
                if len(timed_data_df_truncated) > 0: # the sample has been degraded beyond 50% of initial
                    list_of_t50_from_initial.append(timed_data_df_truncated['Time[s]'][timed_data_df_truncated.first_valid_index()])
                else:
                    list_of_t50_from_initial.append(np.nan)
                
                """Finding t80 from max"""
                boolean_series_t80 = timed_data_df_truncated_at_max['EQE [%]'].ge(0.8*timed_data_df['EQE [%]'][max_eqe_index]) # true for values above 80 %
                if sum(boolean_series_t80) < len(timed_data_df_truncated_at_max): # the sample has been degraded beyond t80 from max
                    t80_from_max_index = max_eqe_index + sum(boolean_series_t80)
                    list_of_t80_from_max.append(timed_data_df['Time[s]'][t80_from_max_index] - timed_data_df['Time[s]'][max_eqe_index])
                else:
                    list_of_t80_from_max.append(np.nan)
                    
                """Finding t80 from initial"""
                truncated_series = timed_data_df['EQE [%]'].le(0.8*timed_data_df['EQE [%]'][timed_data_df.first_valid_index()])
                timed_data_df_truncated = timed_data_df.truncate(before=len(timed_data_df)-sum(truncated_series)) # truncating to ignore values that are above 80% of initial
                if len(timed_data_df_truncated) > 0: # the sample has been degraded beyond 80% of initial
                    list_of_t80_from_initial.append(timed_data_df_truncated['Time[s]'][timed_data_df_truncated.first_valid_index()])
                else:
                    list_of_t80_from_initial.append(np.nan)
        else:
            list_of_durations.append(np.nan)
            list_of_max_eqe.append(np.nan)
            list_of_time_to_max_eqe.append(np.nan)
            list_of_t50_from_max.append(np.nan)
            list_of_t50_from_initial.append(np.nan)
            list_of_t80_from_max.append(np.nan)
            list_of_t80_from_initial.append(np.nan)   
        
    return [list_of_durations, list_of_max_eqe, list_of_time_to_max_eqe, list_of_t50_from_max, list_of_t50_from_initial, list_of_t80_from_max, list_of_t80_from_initial]

"""Plotting data per pixel"""
def plot_single_pixels(series_of_data_file_names,substrate_label,pixel_number):
    array_size = len(series_of_data_file_names)
    colour_indices = np.floor(np.linspace(0,999,array_size))
    colour_indices = colour_indices.astype(int)
    
    min_current = 0
    max_current = 0
    min_voltage = 0
    max_voltage = 0
    min_eqe = 0
    max_eqe = 0
    min_luminance = 0
    max_luminance = 0
    min_radiance = 0
    max_radiance = 0
    
    """EQE(V)"""
    fig_eqe_voltage = plt.figure(1)
    fig_eqe_voltage.set_size_inches(9,8)
    fig_eqe_voltage.set_tight_layout(True)
    ax_eqe_voltage = plt.gca()
    
    """EQE(J)"""
    fig_eqe_current = plt.figure(2)
    fig_eqe_current.set_size_inches(9,8)
    fig_eqe_current.set_tight_layout(True)
    ax_eqe_current = plt.gca()
    
    """EQE(J) logarithmic J"""
    fig_eqe_current_log = plt.figure(3)
    fig_eqe_current_log.set_size_inches(9,8)
    fig_eqe_current_log.set_tight_layout(True)
    ax_eqe_current_log = plt.gca()
    
    """J(V)"""
    fig_current_voltage = plt.figure(4)
    fig_current_voltage.set_size_inches(9,8)
    fig_current_voltage.set_tight_layout(True)
    ax_current_voltage = plt.gca()
    
    """J(V) logarithmic J"""
    fig_current_voltage_log = plt.figure(5)
    fig_current_voltage_log.set_size_inches(9,8)
    fig_current_voltage_log.set_tight_layout(True)
    ax_current_voltage_log = plt.gca()
    
    """R(V)"""
    fig_radiance_voltage = plt.figure(6)
    fig_radiance_voltage.set_size_inches(9,8)
    fig_radiance_voltage.set_tight_layout(True)
    ax_radiance_voltage = plt.gca()
    
    """R(V) logarithmic R"""
    fig_radiance_voltage_log = plt.figure(7)
    fig_radiance_voltage_log.set_size_inches(9,8)
    fig_radiance_voltage_log.set_tight_layout(True)
    ax_radiance_voltage_log = plt.gca()
    
    """R(J)"""
    fig_radiance_current = plt.figure(8)
    fig_radiance_current.set_size_inches(9,8)
    fig_radiance_current.set_tight_layout(True)
    ax_radiance_current = plt.gca()
    
    """JVL (J(V) and L(V)) logarithmic J and L"""
    fig_jvl_log = plt.figure(9)
    fig_jvl_log.set_size_inches(10,8)
    fig_jvl_log.set_tight_layout(True)
    ax_jvl_log = plt.gca()
    ax_jvl_log_twin = ax_jvl_log.twinx()  # instantiate a second axes that shares the same x-axis
    
    """Spectrum(V) for last scan"""
    if g46_integrating_sphere:
        fig_spectrum_voltage = plt.figure(10)
        fig_spectrum_voltage.set_size_inches(9,8)
        fig_spectrum_voltage.set_tight_layout(True)
        ax_spectrum_voltage = plt.gca()
        
    """Spectrum(V) for last scan, normalised"""
    if g46_integrating_sphere:
        fig_spectrum_voltage_norm = plt.figure(11)
        fig_spectrum_voltage_norm.set_size_inches(9,8)
        fig_spectrum_voltage_norm.set_tight_layout(True)
        ax_spectrum_voltage_norm = plt.gca()
    
    """Spectrum(scan) at last measurement point, normalised"""
    if g46_integrating_sphere:
        fig_spectrum_scan_norm = plt.figure(12)
        fig_spectrum_scan_norm.set_size_inches(9,8)
        fig_spectrum_scan_norm.set_tight_layout(True)
        ax_spectrum_scan_norm = plt.gca()
    
    for scan in range(array_size):
        file_path = os.path.join(data_folder_path,series_of_data_file_names.iloc[scan])
        if g44_unprocessed:
            df = pd.read_csv(file_path,delimiter='\t')
            scan_info = fetch_scan_info_unprocessed(series_of_data_file_names.iloc[scan])
        elif g44_processed:
            df = pd.read_csv(file_path)
            scan_info = fetch_scan_info_processed(series_of_data_file_names.iloc[scan])
        elif g46_integrating_sphere:
            df = pd.read_csv(file_path,delimiter='\t')
            filename_spectrum = series_of_data_file_names.iloc[scan][0:-3] + 'spa'
            file_path_spectrum = os.path.join(data_folder_path,filename_spectrum)
            spectrum_df = pd.read_csv(file_path_spectrum,sep='\t',skiprows=7, header=None)
            scan_info = fetch_scan_info_unprocessed(series_of_data_file_names.iloc[scan])
            
        """Flip the sign of current density and voltage data if necessary"""
        if device_nip and not g46_integrating_sphere:
            df['LED voltage [V]'] = -df['LED voltage [V]']
            df['LED current [mA/cm2]'] = -df['LED current [mA/cm2]']
        elif device_pin and g46_integrating_sphere:
            df['Voltage [V]'] = -df['Voltage [V]']
            df['Current [mA/cm2]'] = -df['Current [mA/cm2]']
        
        """Write to plots"""
        if g44_processed or g44_unprocessed:
            ax_eqe_voltage.plot(df['LED voltage [V]'], df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_eqe_current.plot(df['LED current [mA/cm2]'], df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_current_voltage.plot(df['LED voltage [V]'], df['LED current [mA/cm2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_voltage.plot(df['LED voltage [V]'], df['Radiance [W/sr/m2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_current.plot(df['LED current [mA/cm2]'], df['Radiance [W/sr/m2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            
            """Avoiding crash because of negatives in logarithmic plots"""
            if df['LED current [mA/cm2]'].min() > 0:
                non_negative_current = df['LED current [mA/cm2]']
            else:
                non_negative_current = df['LED current [mA/cm2]']
                non_negative_current[non_negative_current < 0] = np.nan
                
            if df['Luminance [cd/m2]'].min() > 0:
                non_negative_luminance = df['Luminance [cd/m2]']
            else:
                non_negative_luminance = df['Luminance [cd/m2]']
                non_negative_luminance[non_negative_luminance < 0] = np.nan
                
            if df['Radiance [W/sr/m2]'].min() > 0:
                non_negative_radiance = df['Radiance [W/sr/m2]']
            else:
                non_negative_radiance = df['Radiance [W/sr/m2]']
                non_negative_radiance[non_negative_radiance < 0] = np.nan
            
            ax_eqe_current_log.semilogx(non_negative_current, df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_jvl_log.semilogy(df['LED voltage [V]'], non_negative_current, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_jvl_log_twin.semilogy(df['LED voltage [V]'], non_negative_luminance,'.-', color=COLOURS_ORANGES[colour_indices[scan]], label=str(scan_info[3]))
            ax_current_voltage_log.semilogy(df['LED voltage [V]'], non_negative_current, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_voltage_log.semilogy(df['LED voltage [V]'], non_negative_radiance, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
                
        elif g46_integrating_sphere:
            ax_eqe_voltage.plot(df['Voltage [V]'], df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_eqe_current.plot(df['Current [mA/cm2]'], df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_current_voltage.plot(df['Voltage [V]'], df['Current [mA/cm2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_voltage.plot(df['Voltage [V]'], df['Radiance [W/sr/m2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_current.plot(df['Current [mA/cm2]'], df['Radiance [W/sr/m2]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            
            """Avoiding crash because of negatives in logarithmic plots"""
            if df['Current [mA/cm2]'].min() > 0:
                non_negative_current = df['Current [mA/cm2]']
            else:
                non_negative_current = df['Current [mA/cm2]']
                non_negative_current[non_negative_current < 0] = np.nan
                
            if df['Luminance [cd/m2]'].min() > 0:
                non_negative_luminance = df['Luminance [cd/m2]']
            else:
                non_negative_luminance = df['Luminance [cd/m2]']
                non_negative_luminance[non_negative_luminance < 0] = np.nan
                
            if df['Radiance [W/sr/m2]'].min() > 0:
                non_negative_radiance = df['Radiance [W/sr/m2]']
            else:
                non_negative_radiance = df['Radiance [W/sr/m2]']
                non_negative_radiance[non_negative_radiance < 0] = np.nan
            
            ax_eqe_current_log.semilogx(non_negative_current, df['EQE [%]'], color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_jvl_log.semilogy(df['Voltage [V]'], non_negative_current, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_jvl_log_twin.semilogy(df['Voltage [V]'], non_negative_luminance,'.-', color=COLOURS_ORANGES[colour_indices[scan]], label=str(scan_info[3]))
            ax_current_voltage_log.semilogy(df['Voltage [V]'], non_negative_current, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            ax_radiance_voltage_log.semilogy(df['Voltage [V]'], non_negative_radiance, color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
                
            """Writing spectrum(scan) plot"""
            num_spectrums = len(spectrum_df.columns) - 1
            if num_spectrums > 0: # avoid crash by not plotting if no intensity data
                ax_spectrum_scan_norm.plot(spectrum_df[0], spectrum_df[spectrum_df.columns[-1]]/spectrum_df[spectrum_df.columns[-1]].max(), color=COLOURS_GREENS[colour_indices[scan]], label=str(scan_info[3]))
            
            """Writing spectrum(V) plots"""
            if scan == array_size - 1: # only for last scan of the pixel
                num_spectrums = len(spectrum_df.columns) - 1
                num_spectrums_to_plot = 6 # must be less than or equal to num_spectrum
                colour_indices_spectrum = np.floor(np.linspace(0,999,num_spectrums_to_plot))
                colour_indices_spectrum = colour_indices_spectrum.astype(int)
                if num_spectrums_to_plot <= num_spectrums: # only plot if enough spectrums in file
                    interval = np.floor(num_spectrums/num_spectrums_to_plot)    
                    spectrum_indices_list = np.linspace(num_spectrums-interval*(num_spectrums_to_plot-1),num_spectrums,num_spectrums_to_plot)
                    
                    for i in range(num_spectrums_to_plot):
                        label_string = '%1.2f' % df['Voltage [V]'][spectrum_indices_list[i]-1] + ' V'
                        ax_spectrum_voltage.plot(spectrum_df[0], spectrum_df[spectrum_indices_list[i]], color=COLOURS_GREENS[colour_indices_spectrum[i]], label=label_string)
                        ax_spectrum_voltage_norm.plot(spectrum_df[0], spectrum_df[spectrum_indices_list[i]]/spectrum_df[spectrum_indices_list[i]].max(), color=COLOURS_GREENS[colour_indices_spectrum[i]], label=label_string)
                    
        """Find axis limits"""
        if g44_unprocessed or g44_processed:        
            if min_current > df['LED current [mA/cm2]'].min() or min_current == 0 or np.isnan(min_current):
                min_current = df['LED current [mA/cm2]'].min()
            
            if max_current < df['LED current [mA/cm2]'].max() or max_current == 0 or np.isnan(max_current):
                max_current = df['LED current [mA/cm2]'].max()
                
            if min_voltage > df['LED voltage [V]'].min() or min_voltage == 0 or np.isnan(min_voltage):
                min_voltage = df['LED voltage [V]'].min()
                
            if max_voltage < df['LED voltage [V]'].max() or max_voltage == 0 or np.isnan(max_voltage):
                max_voltage = df['LED voltage [V]'].max()
        elif g46_integrating_sphere:
            if min_current > df['Current [mA/cm2]'].min() or min_current == 0 or np.isnan(min_current):
                min_current = df['Current [mA/cm2]'].min()
            
            if max_current < df['Current [mA/cm2]'].max() or max_current == 0 or np.isnan(max_current):
                max_current = df['Current [mA/cm2]'].max()
                
            if min_voltage > df['Voltage [V]'].min() or min_voltage == 0 or np.isnan(min_voltage):
                min_voltage = df['Voltage [V]'].min()
                
            if max_voltage < df['Voltage [V]'].max() or max_voltage == 0 or np.isnan(max_voltage):
                max_voltage = df['Voltage [V]'].max()
                
        if min_eqe > df['EQE [%]'].min() or min_eqe == 0 or np.isnan(min_eqe):
            min_eqe = df['EQE [%]'].min()
            
        if max_eqe < df['EQE [%]'].max() or max_eqe == 0 or np.isnan(max_eqe):
            max_eqe = df['EQE [%]'].max()
            
        if min_luminance > df['Luminance [cd/m2]'].min() or min_luminance == 0 or np.isnan(min_luminance):
            min_luminance = df['Luminance [cd/m2]'].min()
            
        if max_luminance < df['Luminance [cd/m2]'].max() or max_luminance == 0 or np.isnan(max_luminance):
            max_luminance = df['Luminance [cd/m2]'].max()
            
        if min_radiance > df['Radiance [W/sr/m2]'].min() or min_radiance == 0 or np.isnan(min_radiance):
            min_radiance = df['Radiance [W/sr/m2]'].min()
            
        if max_radiance < df['Radiance [W/sr/m2]'].max() or max_radiance == 0 or np.isnan(max_radiance):
            max_radiance = df['Radiance [W/sr/m2]'].max()
        
    """Labels and titles""" 
    title = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    
    ax_eqe_voltage.set_title(title)
    ax_eqe_voltage.legend(title='Scan #')
    ax_eqe_voltage.set_xlabel('Voltage [V]')
    ax_eqe_voltage.set_ylabel('EQE [%]')
    
    ax_eqe_current.set_title(title)
    ax_eqe_current.legend(title='Scan #')
    ax_eqe_current.set_xlabel('Current Density [mA/cm2]')
    ax_eqe_current.set_ylabel('EQE [%]')    
    
    ax_eqe_current_log.set_title(title)
    ax_eqe_current_log.legend(title='Scan #')
    ax_eqe_current_log.set_xlabel('Current Density [mA/cm2]')
    ax_eqe_current_log.set_ylabel('EQE [%]')    
    
    ax_current_voltage.set_title(title)
    ax_current_voltage.legend(title='Scan #')
    ax_current_voltage.set_xlabel('Voltage [V]')
    ax_current_voltage.set_ylabel('Current Density [mA/cm2]')
    
    ax_current_voltage_log.set_title(title)
    ax_current_voltage_log.legend(title='Scan #')
    ax_current_voltage_log.set_xlabel('Voltage [V]')
    ax_current_voltage_log.set_ylabel('Current Density [mA/cm2]')
    
    ax_radiance_voltage.set_title(title)
    ax_radiance_voltage.legend(title='Scan #')
    ax_radiance_voltage.set_xlabel('Voltage [V]')
    ax_radiance_voltage.set_ylabel('Radiance [W/sr/m2]')
    
    ax_radiance_voltage_log.set_title(title)
    ax_radiance_voltage_log.legend(title='Scan #')
    ax_radiance_voltage_log.set_xlabel('Voltage [V]')
    ax_radiance_voltage_log.set_ylabel('Radiance [W/sr/m2]')
    
    ax_radiance_current.set_title(title)
    ax_radiance_current.legend(title='Scan #')
    ax_radiance_current.set_xlabel('Current Density [mA/cm2]')
    ax_radiance_current.set_ylabel('Radiance [W/sr/m2]')
    
    ax_jvl_log.set_title(title)
    ax_jvl_log.legend(title='J, Scan #',loc='upper left')
    ax_jvl_log_twin.legend(title='L, Scan #',loc='lower right')
    ax_jvl_log.set_xlabel('Voltage [V]')
    ax_jvl_log.set_ylabel('Current Density [mA/cm2]')    
    ax_jvl_log_twin.set_ylabel('Luminance [cd/m2]')    
    
    if g46_integrating_sphere:
        title_legend = 'Scan # ' + str(scan_info[3]) # this will always be the last scan of the pixel
        ax_spectrum_voltage.set_title(title)
        ax_spectrum_voltage.legend(title=title_legend)
        ax_spectrum_voltage.set_xlabel('Wavelength [nm]')
        ax_spectrum_voltage.set_ylabel('Intensity [counts]')    
    
        ax_spectrum_voltage_norm.set_title(title)
        ax_spectrum_voltage_norm.legend(title=title_legend)
        ax_spectrum_voltage_norm.set_xlabel('Wavelength [nm]')
        ax_spectrum_voltage_norm.set_ylabel('Normalized intensity')    
        
        ax_spectrum_scan_norm.set_title(title)
        ax_spectrum_scan_norm.legend(title='Scan #')
        ax_spectrum_scan_norm.set_xlabel('Wavelength [nm]')
        ax_spectrum_scan_norm.set_ylabel('Normalized intensity')
    
    """Set axis limits and spacing"""
    """Y-axis"""
    if max_eqe > 20:
        ax_eqe_voltage.set_ylim(bottom=0, top=20)
        ax_eqe_current.set_ylim(bottom=0, top=20)
        ax_eqe_current_log.set_ylim(bottom=0, top=20)
    else:
        ax_eqe_voltage.set_ylim(bottom=0, top=np.ceil(max_eqe))
        ax_eqe_current.set_ylim(bottom=0, top=np.ceil(max_eqe))
        ax_eqe_current_log.set_ylim(bottom=0, top=np.ceil(max_eqe))
    
    ax_radiance_voltage_log.set_ylim(bottom=1e-4)
    ax_jvl_log_twin.set_ylim(bottom=1e-1, top=10*max_luminance)
    if g46_integrating_sphere:
        ax_spectrum_voltage.set_ylim(bottom=0)
        ax_spectrum_voltage_norm.set_ylim(bottom=0)
        ax_spectrum_scan_norm.set_ylim(bottom=0)
    
    """X-axis"""
    ax_eqe_voltage.set_xlim(right=np.ceil(max_voltage))
    ax_eqe_current.set_xlim(left=0, right=np.ceil(max_current))
    ax_eqe_current_log.set_xlim(left = 1e-2, right=np.ceil(max_current))
    if g46_integrating_sphere:
        ax_spectrum_voltage.set_xlim(left = 400, right = 700)
        ax_spectrum_voltage_norm.set_xlim(left = 400, right = 700)
        ax_spectrum_scan_norm.set_xlim(left = 400, right = 700)
    
    #ax_eqe_voltage.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
    #ax_current_voltage.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
    #ax_current_voltage_log.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
    #ax_radiance_voltage.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
    #ax_jvl_log.xaxis.set_major_locator(mpl.ticker.MultipleLocator(0.5))
    
    """Graphical details"""
    scalar_linewidth = 2
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_eqe_voltage.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_voltage.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_voltage.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_voltage.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_eqe_current.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_current.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_current.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_current.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_eqe_current_log.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_current_log.xaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_current_log.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_current_log.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_current_voltage.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_voltage.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_voltage.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_voltage.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_current_voltage_log.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_voltage_log.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_voltage_log.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_voltage_log.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    
    ax_radiance_voltage.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_voltage.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_voltage.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_voltage.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_radiance_voltage_log.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_voltage_log.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_voltage_log.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_voltage_log.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    
    ax_radiance_current.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_current.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_current.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_current.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_jvl_log.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_jvl_log.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_jvl_log.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_jvl_log.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_jvl_log_twin.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_jvl_log_twin.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    
    if g46_integrating_sphere:
        ax_spectrum_voltage.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_voltage.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        ax_spectrum_voltage.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_voltage.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
        ax_spectrum_voltage_norm.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_voltage_norm.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        ax_spectrum_voltage_norm.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_voltage_norm.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
        ax_spectrum_scan_norm.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_scan_norm.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        ax_spectrum_scan_norm.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
        ax_spectrum_scan_norm.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
    """Exporting plots"""
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' EQE(V)'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_eqe_voltage.savefig(pdf_path)
    fig_eqe_voltage.savefig(svg_path)
    plt.close(fig_eqe_voltage)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' EQE(J)'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_eqe_current.savefig(pdf_path)
    fig_eqe_current.savefig(svg_path)
    plt.close(fig_eqe_current)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' EQE(J) semi-logarithmic'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_eqe_current_log.savefig(pdf_path)
    fig_eqe_current_log.savefig(svg_path)
    plt.close(fig_eqe_current_log)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' J(V)'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_current_voltage.savefig(pdf_path)
    fig_current_voltage.savefig(svg_path)
    plt.close(fig_current_voltage)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' J(V) semi-logarithmic'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_current_voltage_log.savefig(pdf_path)
    fig_current_voltage_log.savefig(svg_path)
    plt.close(fig_current_voltage_log)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' radiance(V)'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_radiance_voltage.savefig(pdf_path)
    fig_radiance_voltage.savefig(svg_path)
    plt.close(fig_radiance_voltage)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' radiance(V) semi-logarithmic'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_radiance_voltage_log.savefig(pdf_path)
    fig_radiance_voltage_log.savefig(svg_path)
    plt.close(fig_radiance_voltage_log)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' radiance(J)'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_radiance_current.savefig(pdf_path)
    fig_radiance_current.savefig(svg_path)
    plt.close(fig_radiance_current)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
    plot_name = ' jvl'
    export_name_pdf = export_name + plot_name + '.pdf'
    export_name_svg = export_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
    fig_jvl_log.savefig(pdf_path)
    fig_jvl_log.savefig(svg_path)
    plt.close(fig_jvl_log)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    if g46_integrating_sphere:
        export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
        plot_name = ' spectrum(V)'
        export_name_pdf = export_name + plot_name + '.pdf'
        export_name_svg = export_name + plot_name + '.svg'
        pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
        svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
        fig_spectrum_voltage.savefig(pdf_path)
        fig_spectrum_voltage.savefig(svg_path)
        plt.close(fig_spectrum_voltage)
        write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
        
        export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
        plot_name = ' spectrum(V) normalized'
        export_name_pdf = export_name + plot_name + '.pdf'
        export_name_svg = export_name + plot_name + '.svg'
        pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
        svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
        fig_spectrum_voltage_norm.savefig(pdf_path)
        fig_spectrum_voltage_norm.savefig(svg_path)
        plt.close(fig_spectrum_voltage_norm)
        write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
        
        export_name = 'Substrate %s, pixel %i' % (str(substrate_label), pixel_number)
        plot_name = ' spectrum(scan) normalized'
        export_name_pdf = export_name + plot_name + '.pdf'
        export_name_svg = export_name + plot_name + '.svg'
        pdf_path = os.path.join(subfolder_pixel_plots_path,export_name_pdf)
        svg_path = os.path.join(subfolder_pixel_plots_path,export_name_svg)
        fig_spectrum_scan_norm.savefig(pdf_path)
        fig_spectrum_scan_norm.savefig(svg_path)
        plt.close(fig_spectrum_scan_norm)
        write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_pixel_plots_path))
    
    print('Exported .pdf and .svg plots for: %s' % export_name)
    return

"""Plotting summary plots for the scan analysis"""
def plot_scan_summary(scan_summary_df,experiment_name):
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box swarm plot general graphic settings"""
    boxcolour = COLOURS_GREENS[700]
    boxprops = {'edgecolor': boxcolour, 'linewidth': scalar_linewidth, 'facecolor': 'w'}
    lineprops = {'color': boxcolour, 'linewidth': scalar_linewidth}
    boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                           'whiskerprops': lineprops, 'capprops': lineprops,
                           'width': 0.75})
    
    """Plotting all the box swarm plots"""
    """Peak EQE"""
    fig_peak_eqe = plt.figure(1)
    fig_peak_eqe.set_size_inches(9,8)
    fig_peak_eqe.set_tight_layout(True)
    ax_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Peak EQE [%]'].notna()], y=scan_summary_df['Peak EQE [%]'][scan_summary_df['Peak EQE [%]'].notna()], ax=ax_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Peak EQE [%]'].notna()], y=scan_summary_df['Peak EQE [%]'][scan_summary_df['Peak EQE [%]'].notna()], ax=ax_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_peak_eqe.set_xlabel('Substrate label')
    ax_peak_eqe.set_ylabel('Peak EQE [%]')
    ax_peak_eqe.set_title(experiment_name)
    ax_peak_eqe.set_ylim(bottom=0)
    if scan_summary_df['Peak EQE [%]'].max() > 25:
        ax_peak_eqe.set_ylim(top=25)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
        
    """Voltage @ Peak EQE"""
    fig_voltage_at_peak_eqe = plt.figure(2)
    fig_voltage_at_peak_eqe.set_size_inches(9,8)
    fig_voltage_at_peak_eqe.set_tight_layout(True)
    ax_voltage_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], y=scan_summary_df['Voltage at Peak EQE [V]'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], ax=ax_voltage_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], y=scan_summary_df['Voltage at Peak EQE [V]'][scan_summary_df['Voltage at Peak EQE [V]'].notna()], ax=ax_voltage_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_peak_eqe.set_ylim(bottom=0)
    ax_voltage_at_peak_eqe.set_xlabel('Substrate label')
    ax_voltage_at_peak_eqe.set_ylabel('Voltage at Peak EQE [V]')
    ax_voltage_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Peak EQE"""
    fig_current_at_peak_eqe = plt.figure(3)
    fig_current_at_peak_eqe.set_size_inches(9,8)
    fig_current_at_peak_eqe.set_tight_layout(True)
    ax_current_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak EQE [mA/cm2]'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], ax=ax_current_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak EQE [mA/cm2]'][scan_summary_df['Current Density at Peak EQE [mA/cm2]'].notna()], ax=ax_current_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_peak_eqe.set_ylim(bottom=0)
    ax_current_at_peak_eqe.set_xlabel('Substrate label')
    ax_current_at_peak_eqe.set_ylabel('Current Density at Peak EQE [mA/cm2]')
    ax_current_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Peak EQE"""
    fig_radiance_at_peak_eqe = plt.figure(4)
    fig_radiance_at_peak_eqe.set_size_inches(9,8)
    fig_radiance_at_peak_eqe.set_tight_layout(True)
    ax_radiance_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], y=scan_summary_df['Radiance at Peak EQE [W/sr/m2]'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], ax=ax_radiance_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], y=scan_summary_df['Radiance at Peak EQE [W/sr/m2]'][scan_summary_df['Radiance at Peak EQE [W/sr/m2]'].notna()], ax=ax_radiance_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_peak_eqe.set_ylim(bottom=0)
    ax_radiance_at_peak_eqe.set_xlabel('Substrate label')
    ax_radiance_at_peak_eqe.set_ylabel('Radiance at Peak EQE [W/sr/m2]')
    ax_radiance_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Scan # @ Peak EQE"""
    fig_scan_at_peak_eqe = plt.figure(5)
    fig_scan_at_peak_eqe.set_size_inches(9,8)
    fig_scan_at_peak_eqe.set_tight_layout(True)
    ax_scan_at_peak_eqe = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Scan # at Peak EQE'].notna()], y=scan_summary_df['Scan # at Peak EQE'][scan_summary_df['Scan # at Peak EQE'].notna()], ax=ax_scan_at_peak_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Scan # at Peak EQE'].notna()], y=scan_summary_df['Scan # at Peak EQE'][scan_summary_df['Scan # at Peak EQE'].notna()], ax=ax_scan_at_peak_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_scan_at_peak_eqe.set_ylim(bottom=0)
    ax_scan_at_peak_eqe.set_xlabel('Substrate label')
    ax_scan_at_peak_eqe.set_ylabel('Scan # at Peak EQE')
    ax_scan_at_peak_eqe.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Luminescence Turn-on (Peak EQE scan)"""
    fig_radiance_at_turn_on_peak_eqe_scan = plt.figure(6)
    fig_radiance_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_radiance_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_radiance_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], ax=ax_radiance_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'].notna()], ax=ax_radiance_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_radiance_at_turn_on_peak_eqe_scan.set_xlabel('Substrate label')
    ax_radiance_at_turn_on_peak_eqe_scan.set_ylabel('Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)')
    ax_radiance_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_radiance_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_radiance_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Luminescence Turn-on (Peak EQE scan)"""
    fig_voltage_at_turn_on_peak_eqe_scan = plt.figure(7)
    fig_voltage_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_voltage_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_voltage_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], ax=ax_voltage_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'].notna()], ax=ax_voltage_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_voltage_at_turn_on_peak_eqe_scan.set_xlabel('Substrate label')
    ax_voltage_at_turn_on_peak_eqe_scan.set_ylabel('Voltage at Luminescence Turn-on [V] (Peak EQE scan)')
    ax_voltage_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_voltage_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_voltage_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Luminescence Turn-on (Peak EQE scan)"""
    fig_current_at_turn_on_peak_eqe_scan = plt.figure(8)
    fig_current_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_current_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_current_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], ax=ax_current_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'].notna()], ax=ax_current_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_current_at_turn_on_peak_eqe_scan.set_xlabel('Substrate label')
    ax_current_at_turn_on_peak_eqe_scan.set_ylabel('Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)')
    ax_current_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_current_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_current_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Luminescence Turn-on (Peak EQE scan)"""
    fig_eqe_at_turn_on_peak_eqe_scan = plt.figure(9)
    fig_eqe_at_turn_on_peak_eqe_scan.set_size_inches(9,8)
    fig_eqe_at_turn_on_peak_eqe_scan.set_tight_layout(True)
    ax_eqe_at_turn_on_peak_eqe_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], ax=ax_eqe_at_turn_on_peak_eqe_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].notna()], ax=ax_eqe_at_turn_on_peak_eqe_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_turn_on_peak_eqe_scan.set_ylim(bottom=0)
    ax_eqe_at_turn_on_peak_eqe_scan.set_xlabel('Substrate label')
    ax_eqe_at_turn_on_peak_eqe_scan.set_ylabel('EQE at Luminescence Turn-on [%] (Peak EQE scan)')
    ax_eqe_at_turn_on_peak_eqe_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_eqe_at_turn_on_peak_eqe_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_eqe_at_turn_on_peak_eqe_scan.transAxes, rotation='vertical')
    if scan_summary_df['EQE at Luminescence Turn-on [%] (Peak EQE scan)'].max() > 25:
        ax_eqe_at_turn_on_peak_eqe_scan.set_ylim(top=25)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Peak Radiance"""
    fig_peak_radiance = plt.figure(10)
    fig_peak_radiance.set_size_inches(9,8)
    fig_peak_radiance.set_tight_layout(True)
    ax_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], y=scan_summary_df['Peak Radiance [W/sr/m2]'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], ax=ax_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], y=scan_summary_df['Peak Radiance [W/sr/m2]'][scan_summary_df['Peak Radiance [W/sr/m2]'].notna()], ax=ax_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_peak_radiance.set_ylim(bottom=0)
    ax_peak_radiance.set_xlabel('Substrate label')
    ax_peak_radiance.set_ylabel('Peak Radiance [W/sr/m2]')
    ax_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Peak Radiance"""
    fig_voltage_at_peak_radiance = plt.figure(11)
    fig_voltage_at_peak_radiance.set_size_inches(9,8)
    fig_voltage_at_peak_radiance.set_tight_layout(True)
    ax_voltage_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], y=scan_summary_df['Voltage at Peak Radiance [V]'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], ax=ax_voltage_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], y=scan_summary_df['Voltage at Peak Radiance [V]'][scan_summary_df['Voltage at Peak Radiance [V]'].notna()], ax=ax_voltage_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_peak_radiance.set_ylim(bottom=0)
    ax_voltage_at_peak_radiance.set_xlabel('Substrate label')
    ax_voltage_at_peak_radiance.set_ylabel('Voltage at Peak Radiance [V]')
    ax_voltage_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Peak Radiance"""
    fig_current_at_peak_radiance = plt.figure(12)
    fig_current_at_peak_radiance.set_size_inches(9,8)
    fig_current_at_peak_radiance.set_tight_layout(True)
    ax_current_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak Radiance [mA/cm2]'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], ax=ax_current_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], y=scan_summary_df['Current Density at Peak Radiance [mA/cm2]'][scan_summary_df['Current Density at Peak Radiance [mA/cm2]'].notna()], ax=ax_current_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_peak_radiance.set_ylim(bottom=0)
    ax_current_at_peak_radiance.set_xlabel('Substrate label')
    ax_current_at_peak_radiance.set_ylabel('Current Density at Peak Radiance [mA/cm2]')
    ax_current_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Peak Radiance"""
    fig_eqe_at_peak_radiance = plt.figure(13)
    fig_eqe_at_peak_radiance.set_size_inches(9,8)
    fig_eqe_at_peak_radiance.set_tight_layout(True)
    ax_eqe_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], y=scan_summary_df['EQE at Peak Radiance [%]'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], ax=ax_eqe_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], y=scan_summary_df['EQE at Peak Radiance [%]'][scan_summary_df['EQE at Peak Radiance [%]'].notna()], ax=ax_eqe_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_peak_radiance.set_ylim(bottom=0)
    ax_eqe_at_peak_radiance.set_xlabel('Substrate label')
    ax_eqe_at_peak_radiance.set_ylabel('EQE at Peak Radiance [%]')
    ax_eqe_at_peak_radiance.set_title(experiment_name)
    if scan_summary_df['EQE at Peak Radiance [%]'].max() > 25:
        ax_eqe_at_peak_radiance.set_ylim(top=25)    
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Scan # @ Peak Radiance"""
    fig_scan_at_peak_radiance = plt.figure(14)
    fig_scan_at_peak_radiance.set_size_inches(9,8)
    fig_scan_at_peak_radiance.set_tight_layout(True)
    ax_scan_at_peak_radiance = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Scan # at Peak Radiance'].notna()], y=scan_summary_df['Scan # at Peak Radiance'][scan_summary_df['Scan # at Peak Radiance'].notna()], ax=ax_scan_at_peak_radiance, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Scan # at Peak Radiance'].notna()], y=scan_summary_df['Scan # at Peak Radiance'][scan_summary_df['Scan # at Peak Radiance'].notna()], ax=ax_scan_at_peak_radiance, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_scan_at_peak_radiance.set_ylim(bottom=0)
    ax_scan_at_peak_radiance.set_xlabel('Substrate label')
    ax_scan_at_peak_radiance.set_ylabel('Scan # at Peak Radiance')
    ax_scan_at_peak_radiance.set_title(experiment_name)
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Radiance @ Luminescence Turn-on (Peak radiance scan)"""
    fig_radiance_at_turn_on_peak_radiance_scan = plt.figure(15)
    fig_radiance_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_radiance_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_radiance_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], ax=ax_radiance_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], y=scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'][scan_summary_df['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'].notna()], ax=ax_radiance_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_radiance_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_radiance_at_turn_on_peak_radiance_scan.set_xlabel('Substrate label')
    ax_radiance_at_turn_on_peak_radiance_scan.set_ylabel('Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)')
    ax_radiance_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_radiance_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_radiance_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Voltage @ Luminescence Turn-on (Peak radiance scan)"""
    fig_voltage_at_turn_on_peak_radiance_scan = plt.figure(16)
    fig_voltage_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_voltage_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_voltage_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], ax=ax_voltage_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], y=scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'][scan_summary_df['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'].notna()], ax=ax_voltage_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_voltage_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_voltage_at_turn_on_peak_radiance_scan.set_xlabel('Substrate label')
    ax_voltage_at_turn_on_peak_radiance_scan.set_ylabel('Voltage at Luminescence Turn-on [V] (Peak radiance scan)')
    ax_voltage_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_voltage_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_voltage_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Current Density @ Luminescence Turn-on (Peak radiance scan)"""
    fig_current_at_turn_on_peak_radiance_scan = plt.figure(17)
    fig_current_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_current_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_current_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], ax=ax_current_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], y=scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'][scan_summary_df['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'].notna()], ax=ax_current_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_current_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_current_at_turn_on_peak_radiance_scan.set_xlabel('Substrate label')
    ax_current_at_turn_on_peak_radiance_scan.set_ylabel('Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)')
    ax_current_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_current_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_current_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """EQE @ Luminescence Turn-on (Peak radiance scan)"""
    fig_eqe_at_turn_on_peak_radiance_scan = plt.figure(18)
    fig_eqe_at_turn_on_peak_radiance_scan.set_size_inches(9,8)
    fig_eqe_at_turn_on_peak_radiance_scan.set_tight_layout(True)
    ax_eqe_at_turn_on_peak_radiance_scan = plt.gca()
    sns.boxplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], ax=ax_eqe_at_turn_on_peak_radiance_scan, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=scan_summary_df['Substrate label'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], y=scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'][scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].notna()], ax=ax_eqe_at_turn_on_peak_radiance_scan, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=6)
    ax_eqe_at_turn_on_peak_radiance_scan.set_ylim(bottom=0)
    ax_eqe_at_turn_on_peak_radiance_scan.set_xlabel('Substrate label')
    ax_eqe_at_turn_on_peak_radiance_scan.set_ylabel('EQE at Luminescence Turn-on [%] (Peak radiance scan)')
    ax_eqe_at_turn_on_peak_radiance_scan.set_title(experiment_name)
    threshold_string = 'Radiance Threshold: %.1E W/sr/m2' %  radiance_threshold
    ax_eqe_at_turn_on_peak_radiance_scan.text(0.97, 0.05, threshold_string, fontsize=12, transform=ax_eqe_at_turn_on_peak_radiance_scan.transAxes, rotation='vertical')
    if scan_summary_df['EQE at Luminescence Turn-on [%] (Peak radiance scan)'].max() > 25:
        ax_eqe_at_turn_on_peak_radiance_scan.set_ylim(top=25)    
    # The .notna() solution is necessary to avoid an occassional bug with swarmplot/stripplot that made the ylim top default to 1.05e20 for some reason that was likely related to the presence of a certain number of NaNs
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_voltage_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_current_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_radiance_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_scan_at_peak_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_scan_at_peak_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_radiance_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_voltage_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_eqe_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_eqe_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_eqe_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_voltage_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_eqe_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_scan_at_peak_radiance.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_radiance.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_scan_at_peak_radiance.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_scan_at_peak_radiance.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_radiance_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_voltage_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_current_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)       
    
    ax_eqe_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_radiance_scan.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_turn_on_peak_radiance_scan.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)           
    
    """Exporting plots"""
    plot_name = ' Peak EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_peak_eqe.savefig(pdf_path)
    fig_peak_eqe.savefig(svg_path)
    plt.close(fig_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Voltage at Peak EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_peak_eqe.savefig(pdf_path)
    fig_voltage_at_peak_eqe.savefig(svg_path)
    plt.close(fig_voltage_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Current Density at Peak EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_peak_eqe.savefig(pdf_path)
    fig_current_at_peak_eqe.savefig(svg_path)
    plt.close(fig_current_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Radiance at Peak EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_peak_eqe.savefig(pdf_path)
    fig_radiance_at_peak_eqe.savefig(svg_path)
    plt.close(fig_radiance_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Scan # at Peak EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_scan_at_peak_eqe.savefig(pdf_path)
    fig_scan_at_peak_eqe.savefig(svg_path)
    plt.close(fig_scan_at_peak_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Radiance at Luminescence Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_radiance_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_radiance_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Voltage at Luminescence Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_voltage_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_voltage_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Current Density at Luminescence Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_current_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_current_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' EQE at Luminescence Turn-on (Peak EQE scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_turn_on_peak_eqe_scan.savefig(pdf_path)
    fig_eqe_at_turn_on_peak_eqe_scan.savefig(svg_path)
    plt.close(fig_eqe_at_turn_on_peak_eqe_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Peak Radiance box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_peak_radiance.savefig(pdf_path)
    fig_peak_radiance.savefig(svg_path)
    plt.close(fig_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Voltage at Peak Radiance box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_peak_radiance.savefig(pdf_path)
    fig_voltage_at_peak_radiance.savefig(svg_path)
    plt.close(fig_voltage_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Current Density at Peak Radiance box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_peak_radiance.savefig(pdf_path)
    fig_current_at_peak_radiance.savefig(svg_path)
    plt.close(fig_current_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' EQE at Peak Radiance box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_peak_radiance.savefig(pdf_path)
    fig_eqe_at_peak_radiance.savefig(svg_path)
    plt.close(fig_eqe_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Scan # at Peak Radiance box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_scan_at_peak_radiance.savefig(pdf_path)
    fig_scan_at_peak_radiance.savefig(svg_path)
    plt.close(fig_scan_at_peak_radiance)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Radiance at Luminescence Turn-on (Peak Radiance scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_radiance_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_radiance_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_radiance_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Voltage at Luminescence Turn-on (Peak Radiance scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_voltage_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_voltage_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_voltage_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' Current Density at Luminescence Turn-on (Peak Radiance scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_current_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_current_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_current_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    plot_name = ' EQE at Luminescence Turn-on (Peak Radiance scan) box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_eqe_at_turn_on_peak_radiance_scan.savefig(pdf_path)
    fig_eqe_at_turn_on_peak_radiance_scan.savefig(svg_path)
    plt.close(fig_eqe_at_turn_on_peak_radiance_scan)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))    
    
    print('Exported summary .pdf and .svg plots for: %s' % experiment_name)

"""Plotting big data plots for scan analysis"""
def plot_big_data(experiment_name, experiment_data, big_data_df):
    # The violinplots are sometimes very narrow, which I think is related to this: https://stackoverflow.com/a/56357825
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box swarm plot general graphic settings"""
    violincolour = COLOURS_GREENS[600]
    
    """Plotting all the violinplots"""
    """Radiance at Max EQE"""
    fig_radiance_at_max_eqe = plt.figure(1)
    fig_radiance_at_max_eqe.set_size_inches(12,8)
    fig_radiance_at_max_eqe.set_tight_layout(True)
    ax_radiance_at_max_eqe = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at Max EQE [W/sr/m2]'], ax=ax_radiance_at_max_eqe, color=violincolour, inner=None)
    ax_radiance_at_max_eqe.set_xlabel('Substrate label')
    ax_radiance_at_max_eqe.set_ylabel('Radiance at Max EQE [W/sr/m2]')
    ax_radiance_at_max_eqe.set_title(experiment_name)
    ax_radiance_at_max_eqe.set_ylim(bottom=0)
    
    """Voltage at Max EQE"""
    fig_voltage_at_max_eqe = plt.figure(2)
    fig_voltage_at_max_eqe.set_size_inches(12,8)
    fig_voltage_at_max_eqe.set_tight_layout(True)
    ax_voltage_at_max_eqe = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Voltage at Max EQE [V]'], ax=ax_voltage_at_max_eqe, color=violincolour, inner=None)
    ax_voltage_at_max_eqe.set_xlabel('Substrate label')
    ax_voltage_at_max_eqe.set_ylabel('Voltage at Max EQE [V]')
    ax_voltage_at_max_eqe.set_title(experiment_name)
    ax_voltage_at_max_eqe.set_ylim(bottom=0)
    
    """Current Density at Max EQE"""
    fig_current_at_max_eqe = plt.figure(3)
    fig_current_at_max_eqe.set_size_inches(12,8)
    fig_current_at_max_eqe.set_tight_layout(True)
    ax_current_at_max_eqe = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Current Density at Max EQE [mA/cm2]'], ax=ax_current_at_max_eqe, color=violincolour, inner=None)
    ax_current_at_max_eqe.set_xlabel('Substrate label')
    ax_current_at_max_eqe.set_ylabel('Current Density at Max EQE [mA/cm2]')
    ax_current_at_max_eqe.set_title(experiment_name)
    ax_current_at_max_eqe.set_ylim(bottom=0)    
    
    """EQE at Max EQE"""
    fig_eqe_at_max_eqe = plt.figure(4)
    fig_eqe_at_max_eqe.set_size_inches(12,8)
    fig_eqe_at_max_eqe.set_tight_layout(True)
    ax_eqe_at_max_eqe = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['EQE at Max EQE [%]'], ax=ax_eqe_at_max_eqe, color=violincolour, inner=None)
    ax_eqe_at_max_eqe.set_xlabel('Substrate label')
    ax_eqe_at_max_eqe.set_ylabel('EQE at Max EQE [%]')
    ax_eqe_at_max_eqe.set_title(experiment_name)
    ax_eqe_at_max_eqe.set_ylim(bottom=0)
    
    """Radiance at 2.0 V"""
    fig_radiance_at_2p0_V = plt.figure(5)
    fig_radiance_at_2p0_V.set_size_inches(12,8)
    fig_radiance_at_2p0_V.set_tight_layout(True)
    ax_radiance_at_2p0_V = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at 2.0 V [W/sr/m2]'], ax=ax_radiance_at_2p0_V, color=violincolour, inner=None)
    ax_radiance_at_2p0_V.set_xlabel('Substrate label')
    ax_radiance_at_2p0_V.set_ylabel('Radiance at 2.0 V [W/sr/m2]')
    ax_radiance_at_2p0_V.set_title(experiment_name)
    ax_radiance_at_2p0_V.set_ylim(bottom=0)
    
    """Radiance at 2.5 V"""
    fig_radiance_at_2p5_V = plt.figure(6)
    fig_radiance_at_2p5_V.set_size_inches(12,8)
    fig_radiance_at_2p5_V.set_tight_layout(True)
    ax_radiance_at_2p5_V = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at 2.5 V [W/sr/m2]'], ax=ax_radiance_at_2p5_V, color=violincolour, inner=None)
    ax_radiance_at_2p5_V.set_xlabel('Substrate label')
    ax_radiance_at_2p5_V.set_ylabel('Radiance at 2.5 V [W/sr/m2]')
    ax_radiance_at_2p5_V.set_title(experiment_name)
    ax_radiance_at_2p5_V.set_ylim(bottom=0)
        
    """Radiance at 3.0 V"""
    fig_radiance_at_3p0_V = plt.figure(7)
    fig_radiance_at_3p0_V.set_size_inches(12,8)
    fig_radiance_at_3p0_V.set_tight_layout(True)
    ax_radiance_at_3p0_V = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at 3.0 V [W/sr/m2]'], ax=ax_radiance_at_3p0_V, color=violincolour, inner=None)
    ax_radiance_at_3p0_V.set_xlabel('Substrate label')
    ax_radiance_at_3p0_V.set_ylabel('Radiance at 3.0 V [W/sr/m2]')
    ax_radiance_at_3p0_V.set_title(experiment_name)
    ax_radiance_at_3p0_V.set_ylim(bottom=0) 
    
    """Radiance at 3.5 V"""
    fig_radiance_at_3p5_V = plt.figure(8)
    fig_radiance_at_3p5_V.set_size_inches(12,8)
    fig_radiance_at_3p5_V.set_tight_layout(True)
    ax_radiance_at_3p5_V = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at 3.5 V [W/sr/m2]'], ax=ax_radiance_at_3p5_V, color=violincolour, inner=None)
    ax_radiance_at_3p5_V.set_xlabel('Substrate label')
    ax_radiance_at_3p5_V.set_ylabel('Radiance at 3.5 V [W/sr/m2]')
    ax_radiance_at_3p5_V.set_title(experiment_name)
    ax_radiance_at_3p5_V.set_ylim(bottom=0)
    
    """Radiance at 4.0 V"""
    fig_radiance_at_4p0_V = plt.figure(9)
    fig_radiance_at_4p0_V.set_size_inches(12,8)
    fig_radiance_at_4p0_V.set_tight_layout(True)
    ax_radiance_at_4p0_V = plt.gca()  
    sns.violinplot(x=big_data_df['Substrate label'], y=big_data_df['Radiance at 4.0 V [W/sr/m2]'], ax=ax_radiance_at_4p0_V, color=violincolour, inner=None)
    ax_radiance_at_4p0_V.set_xlabel('Substrate label')
    ax_radiance_at_4p0_V.set_ylabel('Radiance at 4.0 V [W/sr/m2]')
    ax_radiance_at_4p0_V.set_title(experiment_name)
    ax_radiance_at_4p0_V.set_ylim(bottom=0)
    
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_radiance_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_voltage_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_voltage_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_voltage_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_current_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_current_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_current_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_eqe_at_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_eqe_at_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_eqe_at_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_2p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_2p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_2p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_2p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_2p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_3p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_3p5_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p5_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_3p5_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_3p5_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    ax_radiance_at_4p0_V.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_4p0_V.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_radiance_at_4p0_V.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_radiance_at_4p0_V.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)    
    
    """Exporting plots"""
    plot_name = ' Radiance at Max EQE violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_max_eqe.savefig(pdf_path)
    fig_radiance_at_max_eqe.savefig(svg_path)
    plt.close(fig_radiance_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))
    
    plot_name = ' Voltage at Max EQE violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_voltage_at_max_eqe.savefig(pdf_path)
    fig_voltage_at_max_eqe.savefig(svg_path)
    plt.close(fig_voltage_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Current at Max EQE violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_current_at_max_eqe.savefig(pdf_path)
    fig_current_at_max_eqe.savefig(svg_path)
    plt.close(fig_current_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' EQE at Max EQE violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_eqe_at_max_eqe.savefig(pdf_path)
    fig_eqe_at_max_eqe.savefig(svg_path)
    plt.close(fig_eqe_at_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Radiance at 2p0 V violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_2p0_V.savefig(pdf_path)
    fig_radiance_at_2p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_2p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Radiance at 2p5 V violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_2p5_V.savefig(pdf_path)
    fig_radiance_at_2p5_V.savefig(svg_path)
    plt.close(fig_radiance_at_2p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Radiance at 3p0 V violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_3p0_V.savefig(pdf_path)
    fig_radiance_at_3p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Radiance at 3p5 V violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_3p5_V.savefig(pdf_path)
    fig_radiance_at_3p5_V.savefig(svg_path)
    plt.close(fig_radiance_at_3p5_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    plot_name = ' Radiance at 4p0 V violin plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_big_data_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_big_data_plots_path,export_name_svg)
    fig_radiance_at_4p0_V.savefig(pdf_path)
    fig_radiance_at_4p0_V.savefig(svg_path)
    plt.close(fig_radiance_at_4p0_V)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_big_data_plots_path))    
    
    print('Exported big data .pdf and .svg plots for: %s' % experiment_name)  

"""Plotting summary plots for the timed analysis"""
def plot_timed_data(experiment_data,experiment_name,timed_summary_df):
    """Some constants for graphical settings"""
    scalar_ticksize_major = 5
    scalar_ticksize_minor = 0
    scalar_tickwidth_minor = 0
    scalar_linewidth = 2
    
    """Box swarm plot general graphic settings"""
    boxcolour = COLOURS_GREENS[700]
    boxprops = {'edgecolor': boxcolour, 'linewidth': scalar_linewidth, 'facecolor': 'w'}
    lineprops = {'color': boxcolour, 'linewidth': scalar_linewidth}
    boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                           'whiskerprops': lineprops, 'capprops': lineprops,
                           'width': 0.75})
    
    series_of_data_file_names = experiment_data['File name (timed)']
    array_size = len(series_of_data_file_names)
    colour_indices = np.floor(np.linspace(0,999,array_size))   
    colour_indices = colour_indices.astype(int)
    
    """Plotting all the timed measurements in full in various plots"""
    """EQE(t)"""
    fig_timed = plt.figure(1)
    fig_timed.set_size_inches(11,8)
    fig_timed.set_tight_layout(True)
    ax_timed = plt.gca()
    
    """VTL (V(t) and L(t))"""
    fig_vtl = plt.figure(2)
    fig_vtl.set_size_inches(12,8)
    fig_vtl.set_tight_layout(True)
    ax_vtl = plt.gca()
    ax_vtl_twin = ax_vtl.twinx()  # instantiate a second axes that shares the same x-axis
    
    """VTL (V(t) and L(t)) logarithmic V and L"""
    fig_vtl_log = plt.figure(3)
    fig_vtl_log.set_size_inches(12,8)
    fig_vtl_log.set_tight_layout(True)
    ax_vtl_log = plt.gca()
    ax_vtl_log_twin = ax_vtl_log.twinx()  # instantiate a second axes that shares the same x-axis
    
    for scan in range(array_size):
        file_path = os.path.join(data_folder_path,series_of_data_file_names.iloc[scan])
        if g44_unprocessed or g46_integrating_sphere:
            scan_df = pd.read_csv(file_path,delimiter='\t')
            scan_info = fetch_scan_info_unprocessed(series_of_data_file_names.iloc[scan])
        elif g44_processed:
            scan_df = pd.read_csv(file_path)
            scan_info = fetch_scan_info_processed(series_of_data_file_names.iloc[scan])
        
        label_string = str(scan_info[1]) + ' p. ' + str(scan_info[2]) + ' s. ' + str(scan_info[3])
        ax_timed.plot(scan_df['Time[s]'], scan_df['EQE [%]'], color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
        if g44_unprocessed or g44_processed:
            ax_vtl.plot(scan_df['Time[s]'], scan_df['LED Voltage [V]'], color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            ax_vtl_twin.plot(scan_df['Time[s]'], scan_df['Luminance [cd/m2]'], '.-', markevery=10, color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            if scan_df['LED Voltage [V]'].min() > 0: # avoiding crash because of negatives
                ax_vtl_log.semilogy(scan_df['Time[s]'], scan_df['LED Voltage [V]'], color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
                ax_vtl_log_twin.semilogy(scan_df['Time[s]'], scan_df['Luminance [cd/m2]'], '.-', markevery=10, color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            else:
                print('Not plotting semilog plots with this scan (negative values): substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                write_to_log('Not plotting semilog plots with this scan (negative values): substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
        elif g46_integrating_sphere:
            ax_vtl.plot(scan_df['Time[s]'], scan_df['Voltage [V]'], color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            ax_vtl_twin.plot(scan_df['Time[s]'], scan_df['Luminance [cd/m2]'], '.-', markevery=10, color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            if scan_df['Voltage [V]'].min() > 0: # avoiding crash because of negatives
                ax_vtl_log.semilogy(scan_df['Time[s]'], scan_df['Voltage [V]'], color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
                ax_vtl_log_twin.semilogy(scan_df['Time[s]'], scan_df['Luminance [cd/m2]'], '.-', markevery=10, color=COLOURS_SPECTRAL[colour_indices[scan]], label=label_string)
            else:
                print('Not plotting semilog plots with this scan (negative values): substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
                write_to_log('Not plotting semilog plots with this scan (negative values): substrate %s pixel %i scan %i.' % (scan_info[1], scan_info[2], scan_info[3]))
        
    ax_timed.set_title(experiment_name)
    ax_timed.legend(title='Sample', bbox_to_anchor=(1, 1), fontsize='xx-small')
    ax_timed.set_xlabel('Time [s]')
    ax_timed.set_ylabel('EQE [%]')
    ax_timed.set_ylim(bottom=0, top=15)
    
    ax_vtl.set_title(experiment_name)
    ax_vtl.legend(title='Sample', bbox_to_anchor=(1.4, 1), fontsize='xx-small')
    ax_vtl.set_xlabel('Time [s]')
    ax_vtl.set_ylabel('Voltage [V]')  
    ax_vtl_twin.set_ylabel('Luminance [cd/m2]')
    ax_vtl.set_ylim(bottom=0, top=6)
    
    ax_vtl_log.set_title(experiment_name)
    ax_vtl_log.legend(title='Sample', bbox_to_anchor=(1.4, 1), fontsize='xx-small')
    ax_vtl_log.set_xlabel('Time [s]')
    ax_vtl_log.set_ylabel('Voltage [V]')
    ax_vtl_log_twin.set_ylabel('Luminance [cd/m2]')
    ax_vtl_log.set_ylim(bottom = 1, top = 1e1)
    [bottom, top] = ax_vtl_log_twin.get_ylim()
    if bottom < 1e1:
        ax_vtl_log_twin.set_ylim(bottom = 1e1)
    
    """Plotting all the box swarm plots"""
    """Max EQE"""
    fig_max_eqe = plt.figure(4)
    fig_max_eqe.set_size_inches(9,8)
    fig_max_eqe.set_tight_layout(True)
    ax_max_eqe = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Max EQE [%]'], ax=ax_max_eqe, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Max EQE [%]'], ax=ax_max_eqe, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_max_eqe.set_xlabel('Substrate label')
    ax_max_eqe.set_ylabel('Max EQE [%]')
    ax_max_eqe.set_title(experiment_name)
    ax_max_eqe.set_ylim(bottom=0)
    if timed_summary_df['Max EQE [%]'].max() > 25:
        ax_max_eqe.set_ylim(top=25)
        
    """Time to reach max EQE"""
    fig_time_to_max = plt.figure(5)
    fig_time_to_max.set_size_inches(9,8)
    fig_time_to_max.set_tight_layout(True)
    ax_time_to_max = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to max EQE [s]'], ax=ax_time_to_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to max EQE [s]'], ax=ax_time_to_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_max.set_xlabel('Substrate label')
    ax_time_to_max.set_ylabel('Time to max EQE [s]')
    ax_time_to_max.set_title(experiment_name)
    ax_time_to_max.set_ylim(bottom=0)
        
    """Time to 50% of max EQE from time of max"""
    fig_time_to_50_max = plt.figure(6)
    fig_time_to_50_max.set_size_inches(9,8)
    fig_time_to_50_max.set_tight_layout(True)
    ax_time_to_50_max = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 50% of max EQE from time of max [s]'], ax=ax_time_to_50_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'][timed_summary_df['Time to 50% of max EQE from time of max [s]'].notna()], y=timed_summary_df['Time to 50% of max EQE from time of max [s]'][timed_summary_df['Time to 50% of max EQE from time of max [s]'].notna()], ax=ax_time_to_50_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_50_max.set_xlabel('Substrate label')
    ax_time_to_50_max.set_ylabel('Time to 50% of max EQE from time of max [s]')
    ax_time_to_50_max.set_title(experiment_name)
    ax_time_to_50_max.set_ylim(bottom=0)    
    # This .notna() solution removes the NaNs before plotting, which somehow solves an issue with edgecolor=boxcolour in the swarmplot for these two plots only with the dummy data. The error was of type "ValueError: Invalid RGBA argument". I think it has something to do with an arraysize related to boxcolour (e.g. setting edgecolor='k' solves it), but since boxcolour is a constant for all the values and works well in all other plots (despite those having NaNs too, but less), I do not understand exactly what goes wrong.
    
    """Time to 50% of intitial EQE from start"""
    fig_time_to_50_initial = plt.figure(7)
    fig_time_to_50_initial.set_size_inches(9,8)
    fig_time_to_50_initial.set_tight_layout(True)
    ax_time_to_50_initial = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 50% of initial EQE from start [s]'], ax=ax_time_to_50_initial, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'][timed_summary_df['Time to 50% of initial EQE from start [s]'].notna()], y=timed_summary_df['Time to 50% of initial EQE from start [s]'][timed_summary_df['Time to 50% of initial EQE from start [s]'].notna()], ax=ax_time_to_50_initial, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_50_initial.set_xlabel('Substrate label')
    ax_time_to_50_initial.set_ylabel('Time to 50% of initial EQE from start [s]')
    ax_time_to_50_initial.set_title(experiment_name)
    ax_time_to_50_initial.set_ylim(bottom=0)
    # This .notna() solution removes the NaNs before plotting, which somehow solves an issue with edgecolor=boxcolour in the swarmplot for these two plots only with the dummy data. The error was of type "ValueError: Invalid RGBA argument". I think it has something to do with an arraysize related to boxcolour (e.g. setting edgecolor='k' solves it), but since boxcolour is a constant for all the values and works well in all other plots (despite those having NaNs too, but less), I do not understand exactly what goes wrong.
    
    """Time to 80% of max EQE from time of max"""
    fig_time_to_80_max = plt.figure(8)
    fig_time_to_80_max.set_size_inches(9,8)
    fig_time_to_80_max.set_tight_layout(True)
    ax_time_to_80_max = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 80% of max EQE from time of max [s]'], ax=ax_time_to_80_max, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 80% of max EQE from time of max [s]'], ax=ax_time_to_80_max, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_80_max.set_xlabel('Substrate label')
    ax_time_to_80_max.set_ylabel('Time to 80% of max EQE from time of max [s]')
    ax_time_to_80_max.set_title(experiment_name)
    ax_time_to_80_max.set_ylim(bottom=0)
        
    """Time to 80% of intitial EQE from start"""
    fig_time_to_80_initial = plt.figure(9)
    fig_time_to_80_initial.set_size_inches(9,8)
    fig_time_to_80_initial.set_tight_layout(True)
    ax_time_to_80_initial = plt.gca()
    sns.boxplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 80% of initial EQE from start [s]'], ax=ax_time_to_80_initial, fliersize=0, **boxplot_kwargs)
    sns.swarmplot(x=timed_summary_df['Substrate label'], y=timed_summary_df['Time to 80% of initial EQE from start [s]'], ax=ax_time_to_80_initial, color='w', edgecolor=boxcolour, linewidth=scalar_linewidth, size=10)
    ax_time_to_80_initial.set_xlabel('Substrate label')
    ax_time_to_80_initial.set_ylabel('Time to 80% of initial EQE from start [s]')
    ax_time_to_80_initial.set_title(experiment_name)
    ax_time_to_80_initial.set_ylim(bottom=0)
      
    """Graphical details"""
    plt.rcParams['font.size'] = 16
    plt.rcParams['axes.titlesize'] = 12
    plt.rcParams['axes.linewidth'] = scalar_linewidth
    
    ax_timed.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_timed.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_timed.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_timed.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_vtl.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_vtl.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_vtl_twin.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl_twin.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_vtl_log.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl_log.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_vtl_log.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl_log.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl_log_twin.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_vtl_log_twin.yaxis.set_tick_params(which='minor', size=scalar_ticksize_major, width=scalar_linewidth)    
    
    ax_max_eqe.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_max_eqe.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_max_eqe.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_max_eqe.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_50_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_50_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
    ax_time_to_50_initial.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_initial.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_50_initial.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_50_initial.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    
    ax_time_to_80_max.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_max.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_80_max.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_max.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
           
    ax_time_to_80_initial.xaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_initial.xaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
    ax_time_to_80_initial.yaxis.set_tick_params(which='major', size=scalar_ticksize_major, width=scalar_linewidth)
    ax_time_to_80_initial.yaxis.set_tick_params(which='minor', size=scalar_ticksize_minor, width=scalar_tickwidth_minor)
        
    """Exporting plots"""
    plot_name = ' Timed plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_timed.savefig(pdf_path)
    fig_timed.savefig(svg_path)
    plt.close(fig_timed)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
        
    plot_name = ' Timed VTL plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_vtl.savefig(pdf_path)
    fig_vtl.savefig(svg_path)
    plt.close(fig_vtl)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed VTL logarithmic plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_vtl_log.savefig(pdf_path)
    fig_vtl_log.savefig(svg_path)
    plt.close(fig_vtl_log)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed Max EQE box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_max_eqe.savefig(pdf_path)
    fig_max_eqe.savefig(svg_path)
    plt.close(fig_max_eqe)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed time-to-max box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_max.savefig(pdf_path)
    fig_time_to_max.savefig(svg_path)
    plt.close(fig_time_to_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed t50-from-max box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_50_max.savefig(pdf_path)
    fig_time_to_50_max.savefig(svg_path)
    plt.close(fig_time_to_50_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed t50-initial box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_50_initial.savefig(pdf_path)
    fig_time_to_50_initial.savefig(svg_path)
    plt.close(fig_time_to_50_initial)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed t80-from-max box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_80_max.savefig(pdf_path)
    fig_time_to_80_max.savefig(svg_path)
    plt.close(fig_time_to_80_max)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))
    
    plot_name = ' Timed t80-initial box-swarm plot'
    export_name_pdf = experiment_name + plot_name + '.pdf'
    export_name_svg = experiment_name + plot_name + '.svg'
    pdf_path = os.path.join(subfolder_summary_plots_path,export_name_pdf)
    svg_path = os.path.join(subfolder_summary_plots_path,export_name_svg)
    fig_time_to_80_initial.savefig(pdf_path)
    fig_time_to_80_initial.savefig(svg_path)
    plt.close(fig_time_to_80_initial)
    write_to_log('Exported plots %s and %s to %s' % (export_name_pdf, export_name_svg, subfolder_summary_plots_path))        
        
    print('Exported timed summary .pdf and .svg plots for: %s' % experiment_name)    
    return

"""Compiler for analysing scan type data on a pixel level"""
def compiler_scan():
    print('Analysing scan type measurements.') 
    write_to_log('Analysing scan type measurements.')
    
    """Finding the right data files."""
    if g44_unprocessed:    
        write_to_log('Calling function file_finder_ilv.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv()     
    elif g44_processed:
        write_to_log('Calling function file_finder_ilv_processed.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv_processed()
    elif g46_integrating_sphere:
        write_to_log('Calling function file_finder_ilv_spa.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv_spa()
        
    """Grouping experiments and preparing dataframe for analysis data"""
    if file_numbers_list[0] < 1:
        print('No experiments found.')
        write_to_log('No experiments found.')
        return
    else:
        groups_of_experiments = table_of_measurement_files_df.groupby('Experiment name') # grouping into experiments
        
        for experiment_name, experiment_group in groups_of_experiments:   
            experiment_data = groups_of_experiments.get_group(experiment_name) # selecting the data for the current experiment
            print('Analysing experiment', experiment_name, '.')
            write_to_log('Analysing experiment %s.' % experiment_name)
            subgroups_of_experiment_data = experiment_data.groupby(['Experiment name', 'Substrate label', 'Pixel']) # grouping into further subgroups
            table_of_analysis_results_df = pd.DataFrame(index=range(file_numbers_list[2]),\
                                      columns=['Experiment name',\
                                               'Substrate label',\
                                                   'Pixel',\
                                                       'Total # of scans',\
                                                           'Measurement date of last scan (or processing date for post-processed scans)',\
                                                               'Peak EQE [%]',\
                                                                   'Voltage at Peak EQE [V]',\
                                                                       'Current Density at Peak EQE [mA/cm2]',\
                                                                           'Radiance at Peak EQE [W/sr/m2]',\
                                                                               'Scan # at Peak EQE',\
                                                                                   'Luminescence Turn-on radiance threshold [W/sr/m2]',\
                                                                                       'Peak Radiance [W/sr/m2]',\
                                                                                           'Voltage at Peak Radiance [V]',\
                                                                                               'Current Density at Peak Radiance [mA/cm2]',\
                                                                                                   'EQE at Peak Radiance [%]',\
                                                                                                       'Scan # at Peak Radiance',\
                                                                                                           'Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)',\
                                                                                                               'Voltage at Luminescence Turn-on [V] (Peak EQE scan)',\
                                                                                                                   'Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)',\
                                                                                                                       'EQE at Luminescence Turn-on [%] (Peak EQE scan)',\
                                                                                                                           'Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)',\
                                                                                                                               'Voltage at Luminescence Turn-on [V] (Peak radiance scan)',\
                                                                                                                                   'Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)',\
                                                                                                                                       'EQE at Luminescence Turn-on [%] (Peak radiance scan)'])
                                                                                                                                 
            counter = 0
            for subgroup_name, subgroup in subgroups_of_experiment_data:
                write_to_log('Analysing substrate %s pixel %s.' % (subgroup_name[1], subgroup_name[2]))
                subgroup_df = subgroups_of_experiment_data.get_group(subgroup_name) # selecting the data for the current pixel
                
                """Sorting the scans in ascending order (avoiding 1 10 11 2 3 etc)"""
                subgroup_df = subgroup_df.sort_values(by=['Scan #'])    
                
                """Fetching the modification date of the data file of the last scan to probe when the measurement was performed or the processing was done"""
                file_date_info = fetch_file_creation_time(subgroup_df['File name (scan)'].iloc[-1])
                measurement_date_last_scan = file_date_info.strftime('%Y-%m-%d, %H:%M:%S')
                
                """Fetching the peak EQE of the set of scans on this pixel"""
                peak_eqe_data = analyse_peak_eqe(subgroup_df['File name (scan)'])
                
                """Fetching the peak radiance of the set of scans on this pixel"""
                peak_radiance_data = analyse_peak_radiance(subgroup_df['File name (scan)'])
            
                # scanindex = subgroup_df.loc[subgroup_df['Scan #']==pixelData[4]].index.item() # this one is useful for later - it finds the index of the dataframe that gives the peak eqe: subgroup_df['File name'][scanindex]
                
                """Adding all the analysis data to the dataframe"""
                table_of_analysis_results_df.loc[counter]['Experiment name'] = subgroup_name[0] # string
                table_of_analysis_results_df.loc[counter]['Substrate label'] = subgroup_name[1] # string
                table_of_analysis_results_df.loc[counter]['Pixel'] = subgroup_name[2] # integer
                table_of_analysis_results_df.loc[counter]['Total # of scans'] = max(subgroup_df['Scan #']) + 1 # integer, adding +1 since scans are zero indexed
                table_of_analysis_results_df.loc[counter]['Measurement date of last scan (or processing date for post-processed scans)'] = measurement_date_last_scan # string
                table_of_analysis_results_df.loc[counter]['Peak EQE [%]'] = peak_eqe_data[0] # float
                table_of_analysis_results_df.loc[counter]['Voltage at Peak EQE [V]'] = peak_eqe_data[1] # float
                table_of_analysis_results_df.loc[counter]['Current Density at Peak EQE [mA/cm2]'] = peak_eqe_data[2] # float
                table_of_analysis_results_df.loc[counter]['Radiance at Peak EQE [W/sr/m2]'] = peak_eqe_data[3] # float
                table_of_analysis_results_df.loc[counter]['Scan # at Peak EQE'] = peak_eqe_data[4] # integer
                table_of_analysis_results_df.loc[counter]['Luminescence Turn-on radiance threshold [W/sr/m2]'] = radiance_threshold # float
                table_of_analysis_results_df.loc[counter]['Peak Radiance [W/sr/m2]'] = peak_radiance_data[0] # float
                table_of_analysis_results_df.loc[counter]['Voltage at Peak Radiance [V]'] = peak_radiance_data[1] # float
                table_of_analysis_results_df.loc[counter]['Current Density at Peak Radiance [mA/cm2]'] = peak_radiance_data[2] # float
                table_of_analysis_results_df.loc[counter]['EQE at Peak Radiance [%]'] = peak_radiance_data[3] # float
                table_of_analysis_results_df.loc[counter]['Scan # at Peak Radiance'] = peak_radiance_data[4] # integer
                table_of_analysis_results_df.loc[counter]['Radiance at Luminescence Turn-on [W/sr/m2] (Peak EQE scan)'] = peak_eqe_data[5][0] # float
                table_of_analysis_results_df.loc[counter]['Voltage at Luminescence Turn-on [V] (Peak EQE scan)'] = peak_eqe_data[5][1] # float
                table_of_analysis_results_df.loc[counter]['Current Density at Luminescence Turn-on [mA/cm2] (Peak EQE scan)'] = peak_eqe_data[5][2] # float
                table_of_analysis_results_df.loc[counter]['EQE at Luminescence Turn-on [%] (Peak EQE scan)'] = peak_eqe_data[5][3] # float
                table_of_analysis_results_df.loc[counter]['Radiance at Luminescence Turn-on [W/sr/m2] (Peak radiance scan)'] = peak_radiance_data[5][0] # float
                table_of_analysis_results_df.loc[counter]['Voltage at Luminescence Turn-on [V] (Peak radiance scan)'] = peak_radiance_data[5][1] # float
                table_of_analysis_results_df.loc[counter]['Current Density at Luminescence Turn-on [mA/cm2] (Peak radiance scan)'] = peak_radiance_data[5][2] # float
                table_of_analysis_results_df.loc[counter]['EQE at Luminescence Turn-on [%] (Peak radiance scan)'] = peak_radiance_data[5][3] # float
                
                counter = counter + 1                
                
                """Plot the data for this pixel"""
                plot_single_pixels(subgroup_df['File name (scan)'],subgroup_name[1],subgroup_name[2])
                # note that plot_single_pixels also plots values that was cleaned away in the analyse_peak_eqe treatment
            
            """Plot values from the analysis of (cleaned) data from analyse_peak_eqe"""
            plot_scan_summary(table_of_analysis_results_df,experiment_name)
    
            """Export the analysis data in table_of_analysis_results_df to .csv"""
            export_name = experiment_name + ' scan measurements analysis summary ' + date.strftime('%Y-%m-%d') + '.csv'
            csv_path = os.path.join(summary_folder_path, export_name)
            table_of_analysis_results_df.to_csv(csv_path,index=False)
            print('Exported scan data summary file (%s) to subfolder: %s.\n' % (export_name, summary_folder_name))
            write_to_log(['----------', 'Exported scan data summary file (%s) to subfolder: %s.' % (export_name, summary_folder_name)])

"""Compiler for analysing scan type data on an individual scan level"""
def compiler_scan_big_data():
    print('Analysing scan type measurements for "big data" collection.') 
    write_to_log('Analysing scan type measurements for "big data" collection.')
    
    """Finding the right data files."""
    if g44_unprocessed:    
        write_to_log('Calling function file_finder_ilv.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv()     
    elif g44_processed:
        write_to_log('Calling function file_finder_ilv_processed.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv_processed()
    elif g46_integrating_sphere:
        write_to_log('Calling function file_finder_ilv_spa.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_ilv_spa()
    
    """Grouping experiments and preparing dataframe for analysis data"""
    if file_numbers_list[0] < 1:
        print('No experiments found.')
        write_to_log('No experiments found.')
        return
    else:
        groups_of_experiments = table_of_measurement_files_df.groupby('Experiment name') # grouping into experiments
        
        """Analysing"""
        for experiment_name, experiment_group in groups_of_experiments:   
            experiment_data = groups_of_experiments.get_group(experiment_name) # selecting the data for the current experiment
            print('Analysing experiment %s with %i files.' % (experiment_name,len(experiment_data)))
            write_to_log('Analysing experiment %s with %i files.' % (experiment_name,len(experiment_data)))
           
            """Pre-allocating"""
            list_of_measurement_times = []
            list_of_measurement_times_string = []
            list_of_radiance_at_max_eqe = []
            list_of_voltage_at_max_eqe = []
            list_of_current_at_max_eqe = []
            list_of_eqe_at_max_eqe = []
            list_of_radiance_at_luminescence_turn_on = []
            list_of_voltage_at_luminescence_turn_on = []
            list_of_current_at_luminescence_turn_on = []
            list_of_eqe_at_luminescence_turn_on = []
            list_of_radiance_at_2_0_V = []
            list_of_voltage_at_2_0_V = []
            list_of_current_at_2_0_V = []
            list_of_eqe_at_2_0_V = []
            list_of_radiance_at_2_5_V = []
            list_of_voltage_at_2_5_V = []
            list_of_current_at_2_5_V = []
            list_of_eqe_at_2_5_V = []
            list_of_radiance_at_3_0_V = []
            list_of_voltage_at_3_0_V = []
            list_of_current_at_3_0_V = []
            list_of_eqe_at_3_0_V = []
            list_of_radiance_at_3_5_V = []
            list_of_voltage_at_3_5_V = []
            list_of_current_at_3_5_V = []
            list_of_eqe_at_3_5_V = []
            list_of_radiance_at_4_0_V = []
            list_of_voltage_at_4_0_V = []
            list_of_current_at_4_0_V = []
            list_of_eqe_at_4_0_V = []
            
            """Getting the values for all the scans"""
            for i in range(len(experiment_data)):
                """Analysing the scan"""
                list_of_max_eqe_related_values = analyse_max_eqe(experiment_data['File name (scan)'][i])
                list_of_luminescence_turn_on_related_values = analyse_luminescence_turn_on(experiment_data['File name (scan)'][i])
                list_of_voltage_related_values = analyse_voltage(experiment_data['File name (scan)'][i])
                measurement_time = fetch_file_creation_time(experiment_data['File name (scan)'][i])
                measurement_time_string = measurement_time.strftime('%Y-%m-%d, %H:%M:%S')
                
                """Appending the values to the lists"""
                list_of_measurement_times.append(measurement_time)
                list_of_measurement_times_string.append(measurement_time_string)
                list_of_radiance_at_max_eqe.append(list_of_max_eqe_related_values[0])
                list_of_voltage_at_max_eqe.append(list_of_max_eqe_related_values[1])
                list_of_current_at_max_eqe.append(list_of_max_eqe_related_values[2])
                list_of_eqe_at_max_eqe.append(list_of_max_eqe_related_values[3])
                list_of_radiance_at_luminescence_turn_on.append(list_of_luminescence_turn_on_related_values[0])
                list_of_voltage_at_luminescence_turn_on.append(list_of_luminescence_turn_on_related_values[1])
                list_of_current_at_luminescence_turn_on.append(list_of_luminescence_turn_on_related_values[2])
                list_of_eqe_at_luminescence_turn_on.append(list_of_luminescence_turn_on_related_values[3])
                list_of_radiance_at_2_0_V.append(list_of_voltage_related_values[0])
                list_of_voltage_at_2_0_V.append(list_of_voltage_related_values[1])
                list_of_current_at_2_0_V.append(list_of_voltage_related_values[2])
                list_of_eqe_at_2_0_V.append(list_of_voltage_related_values[3])
                list_of_radiance_at_2_5_V.append(list_of_voltage_related_values[4])
                list_of_voltage_at_2_5_V.append(list_of_voltage_related_values[5])
                list_of_current_at_2_5_V.append(list_of_voltage_related_values[6])
                list_of_eqe_at_2_5_V.append(list_of_voltage_related_values[7])
                list_of_radiance_at_3_0_V.append(list_of_voltage_related_values[8])
                list_of_voltage_at_3_0_V.append(list_of_voltage_related_values[9])
                list_of_current_at_3_0_V.append(list_of_voltage_related_values[10])
                list_of_eqe_at_3_0_V.append(list_of_voltage_related_values[11])
                list_of_radiance_at_3_5_V.append(list_of_voltage_related_values[12])
                list_of_voltage_at_3_5_V.append(list_of_voltage_related_values[13])
                list_of_current_at_3_5_V.append(list_of_voltage_related_values[14])
                list_of_eqe_at_3_5_V.append(list_of_voltage_related_values[15])
                list_of_radiance_at_4_0_V.append(list_of_voltage_related_values[16])
                list_of_voltage_at_4_0_V.append(list_of_voltage_related_values[17])
                list_of_current_at_4_0_V.append(list_of_voltage_related_values[18])
                list_of_eqe_at_4_0_V.append(list_of_voltage_related_values[19])
                
            """Moving data into dataframe"""
            table_of_big_data_analysis_results_df = pd.DataFrame(data={'Experiment name': table_of_measurement_files_df['Experiment name'],\
                                                                    'Substrate label': table_of_measurement_files_df['Substrate label'],\
                                                                    'Pixel': table_of_measurement_files_df['Pixel'],\
                                                                    'Scan #': table_of_measurement_files_df['Scan #'],\
                                                                    'Measurement time (datetime)': list_of_measurement_times,\
                                                                    'Measurement time (string)': list_of_measurement_times_string,\
                                                                    'Radiance at Max EQE [W/sr/m2]': list_of_radiance_at_max_eqe,\
                                                                    'Voltage at Max EQE [V]': list_of_voltage_at_max_eqe,\
                                                                    'Current Density at Max EQE [mA/cm2]': list_of_current_at_max_eqe,\
                                                                    'EQE at Max EQE [%]': list_of_eqe_at_max_eqe,\
                                                                    'Radiance at Luminescence Turn-On [W/sr/m2]': list_of_radiance_at_luminescence_turn_on,\
                                                                    'Voltage at Luminescence Turn-On [V]': list_of_voltage_at_luminescence_turn_on,\
                                                                    'Current Density at Luminescence Turn-On [mA/cm2]': list_of_current_at_luminescence_turn_on,\
                                                                    'EQE at Luminescence Turn-On [mA/cm2]': list_of_eqe_at_luminescence_turn_on,\
                                                                    'Radiance at 2.0 V [W/sr/m2]': list_of_radiance_at_2_0_V,\
                                                                    'Voltage at 2.0 V [V]': list_of_voltage_at_2_0_V,\
                                                                    'Current Density at 2.0 V [mA/cm2]': list_of_current_at_2_0_V,\
                                                                    'EQE at 2.0 V [%]': list_of_eqe_at_2_0_V,\
                                                                    'Radiance at 2.5 V [W/sr/m2]': list_of_radiance_at_2_5_V,\
                                                                    'Voltage at 2.5 V [V]': list_of_voltage_at_2_5_V,\
                                                                    'Current Density at 2.5 V [mA/cm2]': list_of_current_at_2_5_V,\
                                                                    'EQE at 2.5 V [%]': list_of_eqe_at_2_5_V,\
                                                                    'Radiance at 3.0 V [W/sr/m2]': list_of_radiance_at_3_0_V,\
                                                                    'Voltage at 3.0 V [V]': list_of_voltage_at_3_0_V,\
                                                                    'Current Density at 3.0 V [mA/cm2]': list_of_current_at_3_0_V,\
                                                                    'EQE at 3.0 V [%]': list_of_eqe_at_3_0_V,\
                                                                    'Radiance at 3.5 V [W/sr/m2]': list_of_radiance_at_3_5_V,\
                                                                    'Voltage at 3.5 V [V]': list_of_voltage_at_3_5_V,\
                                                                    'Current Density at 3.5 V [mA/cm2]': list_of_current_at_3_5_V,\
                                                                    'EQE at 3.5 V [%]': list_of_eqe_at_3_5_V,\
                                                                    'Radiance at 4.0 V [W/sr/m2]': list_of_radiance_at_4_0_V,\
                                                                    'Voltage at 4.0 V [V]': list_of_voltage_at_4_0_V,\
                                                                    'Current Density at 4.0 V [mA/cm2]': list_of_current_at_4_0_V,\
                                                                    'EQE at 4.0 V [%]': list_of_eqe_at_4_0_V})
                
            plot_big_data(experiment_name, experiment_data, table_of_big_data_analysis_results_df)
            
            """Export the analysis data in table_of_big_data_analysis_results_df to .csv"""
            export_name = experiment_name + ' scans big data analysis summary ' + date.strftime('%Y-%m-%d') + '.csv'
            csv_path = os.path.join(summary_folder_path, export_name)
            table_of_big_data_analysis_results_df.to_csv(csv_path,index=False)
            print('Exported scans big data summary file (%s) to subfolder: %s.\n' % (export_name, summary_folder_name))
            write_to_log(['----------', 'Exported scans big data summary file (%s) to subfolder: %s.' % (export_name, summary_folder_name)])     

"""Compiler for analysing timed type data"""
def compiler_timed():
    print('Analysing timed type measurements.') 
    write_to_log('Analysing timed type measurements.')
    
    """Finding the right files"""
    if g44_unprocessed:    
        write_to_log('Calling function file_finder_tiv.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_tiv()     
    elif g44_processed:
        write_to_log('Calling function file_finder_tiv_processed.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_tiv_processed()
    elif g46_integrating_sphere:
        write_to_log('Calling function file_finder_tiv_spa.')
        [table_of_measurement_files_df, file_numbers_list] = file_finder_tiv_spa()
    
    """Grouping experiments and preparing dataframe for analysis data"""
    if file_numbers_list[0] < 1:
        print('No experiments found.')
        write_to_log('No experiments found.')
        return
    else:
        groups_of_experiments = table_of_measurement_files_df.groupby('Experiment name') # grouping into experiments
        
        """Analysing"""
        for experiment_name, experiment_group in groups_of_experiments:   
            experiment_data = groups_of_experiments.get_group(experiment_name) # selecting the data for the current experiment
            print('Analysing experiment', experiment_name, '.')
            write_to_log('Analysing experiment %s.' % experiment_name)
            [list_of_durations, list_of_max_eqe, list_of_time_to_max_eqe, list_of_t50_from_max, list_of_t50_from_initial,\
             list_of_t80_from_max, list_of_t80_from_initial] = analyse_timed_key_data(experiment_data['File name (timed)'])
                
            table_of_timed_analysis_results_df = pd.DataFrame(data={'Experiment name': table_of_measurement_files_df['Experiment name'],\
                                                                    'Substrate label': table_of_measurement_files_df['Substrate label'],\
                                                                    'Pixel': table_of_measurement_files_df['Pixel'],\
                                                                    'Scan #': table_of_measurement_files_df['Scan #'],\
                                                                    'Measurement duration [s]': list_of_durations,\
                                                                    'Max EQE [%]': list_of_max_eqe,\
                                                                    'Time to max EQE [s]': list_of_time_to_max_eqe,\
                                                                    'Time to 50% of max EQE from time of max [s]': list_of_t50_from_max,\
                                                                    'Time to 50% of initial EQE from start [s]': list_of_t50_from_initial,\
                                                                    'Time to 80% of max EQE from time of max [s]': list_of_t80_from_max,\
                                                                    'Time to 80% of initial EQE from start [s]': list_of_t80_from_initial})
                
            plot_timed_data(experiment_data, experiment_name, table_of_timed_analysis_results_df)
            
            """Export the analysis data in table_of_timed_analysis_results_df to .csv"""
            export_name = experiment_name + ' timed measurements analysis summary ' + date.strftime('%Y-%m-%d') + '.csv'
            csv_path = os.path.join(summary_folder_path, export_name)
            table_of_timed_analysis_results_df.to_csv(csv_path,index=False)
            print('Exported timed data summary file (%s) to subfolder: %s.\n' % (export_name, summary_folder_name))
            write_to_log(['----------', 'Exported timed data summary file (%s) to subfolder: %s.' % (export_name, summary_folder_name)])            
    
"""Compiling the functions of the program"""
def compiler_main():
    """Collecting user-defined values for these constants and making them global"""   
    global measurement_type
    global file_type_switch
    global architecture_type
    global data_folder_name
    global export_folder_name
    global voltage_threshold
    global realistic_eqe_threshold
    global radiance_threshold
    global noise_threshold
    global time_prompt
    [measurement_type, file_type_switch, architecture_type, data_folder_name, export_folder_name,voltage_threshold,\
     realistic_eqe_threshold, radiance_threshold, noise_threshold] = prompt_user_input()
    time_prompt = datetime.datetime.now()
   
    """"Setting booleans for the different measurement types (scan/timed) and making them global"""
    global type_scan
    global type_timed
    if measurement_type == 1:
        type_scan = True
    elif measurement_type == 2:
        type_scan = False
        
    type_timed = not type_scan
    
    """Setting booleans for the different file origins (measurements setups) and making them global"""
    global g44_unprocessed
    global g44_processed
    global g46_integrating_sphere   
    if file_type_switch == 1: # G44 unprocessed
        g44_unprocessed = True
        g44_processed = False
        g46_integrating_sphere = False
    elif file_type_switch == 2: # G44 processed
        g44_unprocessed = False
        g44_processed = True
        g46_integrating_sphere = False
    elif file_type_switch == 3: # G46 integrating sphere
        g44_unprocessed = False
        g44_processed = False
        g46_integrating_sphere = True
        
    """Setting booleans for the different architecture types and making them global"""
    global device_pin
    global device_nip
    if architecture_type == 1:
        device_pin = True
    elif architecture_type == 2:
        device_pin = False
    
    device_nip = not device_pin
    
    """"Create folder structure and log file"""
    global main_folder_path
    global data_folder_path
    global export_folder_path
    main_folder_path = os.path.dirname(os.path.abspath(__file__)) # find folder of py file
    data_folder_path = os.path.join(main_folder_path,data_folder_name)
    export_folder_path = os.path.join(main_folder_path,export_folder_name) # create subfolder for exporting
    if not os.path.exists(export_folder_path):
        os.mkdir(export_folder_path) # create folder if non-existent
        
    print('\nCreated export folder.')  
    
    global summary_folder_name
    if type_scan:
        summary_folder_name = 'Summary and log files (scans)'
    elif type_timed:
        summary_folder_name = 'Summary and log files (timed)'
    
    global summary_folder_path
    summary_folder_path = os.path.join(export_folder_path,summary_folder_name)
    if not os.path.exists(summary_folder_path):
        os.mkdir(summary_folder_path) # create folder if non-existent
    
    if type_scan:
        global subfolder_pixel_plots_name
        global subfolder_big_data_plots_path
        subfolder_pixel_plots_name = 'Plots per pixel (scans)'
        subfolder_big_data_plots_name = 'Big data plots (scans)'
        global subfolder_pixel_plots_path
        global subfolder_big_data_plots_path
        subfolder_pixel_plots_path = os.path.join(export_folder_path,subfolder_pixel_plots_name)
        subfolder_big_data_plots_path = os.path.join(export_folder_path,subfolder_big_data_plots_name)
        if not os.path.exists(subfolder_pixel_plots_path):
            os.mkdir(subfolder_pixel_plots_path) # create folder if non-existent
        if not os.path.exists(subfolder_big_data_plots_path):
            os.mkdir(subfolder_big_data_plots_path) # create folder if non-existent
        
    global subfolder_summary_plots_name
    if type_scan:
        subfolder_summary_plots_name = 'Summary plots (scans)'
    elif type_timed:
        subfolder_summary_plots_name = 'Summary plots (timed)'
   
    global subfolder_summary_plots_path
    subfolder_summary_plots_path = os.path.join(export_folder_path,subfolder_summary_plots_name)
    if not os.path.exists(subfolder_summary_plots_path):
        os.mkdir(subfolder_summary_plots_path) # create folder if non-existent
    
    print('Created subfolders of export folder.')   
     
    global log_file_path    
    log_file_name = export_folder_name + ' log.txt'
    log_file_path = os.path.join(summary_folder_path,log_file_name)
    log_file = open(log_file_path,'w')
    log_file.close()
    print('Created log file.\n')
    
    """Export initial information to log"""
    strings_to_log_initial = ['Program started: ' + time_start.strftime('%Y-%m-%d, %H:%M:%S'),\
                              'Successful prompt: ' + time_prompt.strftime('%Y-%m-%d, %H:%M:%S') + ' (Elapsed time: ' + str((time_prompt-time_start).total_seconds()) + ' s)',\
                                  '----------',\
                                      'Information prompted:',\
                                          'Measurement type (scans = 1, timed = 2): ' + str(measurement_type),\
                                              'File type switch (G44 unprocessed = 1, G44 processed = 2, G46 integrating sphere = 3): ' + str(file_type_switch),\
                                                  'Architecture type (p-i-n = 1, n-i-p = 2): ' + str(architecture_type),\
                                                      'Data folder name: ' + data_folder_name,\
                                                          'Export folder name: ' + export_folder_name,\
                                                              'Voltage threshold: ' + str(voltage_threshold) + ' V',\
                                                                  'Realistic EQE threshold: ' + str(realistic_eqe_threshold) + ' %',\
                                                                      'Radiance threshold: ' + str(radiance_threshold) + ' W/sr/m2',\
                                                                          'Noise threshold: ' + str(noise_threshold),\
                                                                              '----------',\
                                                                                  'Created export folder: ' + str(export_folder_path),\
                                                                                      'Created subfolder for summary files and the log: ' + str(summary_folder_path),\
                                                                                          '----------']
    write_to_log(strings_to_log_initial)
       
    """Analyse and export data"""
    if type_scan:
        compiler_scan()
        write_to_log('----------')
        compiler_scan_big_data()
    elif type_timed:
        compiler_timed()
    
    global time_end
    time_end = datetime.datetime.now()        
    strings_to_log_end = ['----------',\
                              'Job completed ' + time_end.strftime('%Y-%m-%d, %H:%M:%S') + ' (Elapsed time: ' + str((time_end-time_start).total_seconds()) + ' s)']
    write_to_log(strings_to_log_end)
    print('Job completed.')

compiler_main()

"""Unfinished and suggestion for version 2.x, version 3, or later"""
# EQE as function of radiance or luminance plots in single pixel plots

# Automatic handling and merging of substrates measured in normal and reversed (180 degree rotated) positions

# timestamp plots of when pixels/scans have been measured for scan summary, timed summary, and big data (see attempts below)

# =============================================================================
#     """Measurement time (or processing time)"""
#     fig_measurement_time = plt.figure(2)
#     fig_measurement_time.set_size_inches(9,8)
#     fig_measurement_time.set_tight_layout(True)
#     ax_measurement_time = plt.gca()  
#     sns.violinplot(x=big_data_df['Measurement time'], y=big_data_df['Substrate label'], ax=ax_measurement_time, color=violincolour, inner=None)
#     ax_measurement_time.set_xlabel('Measurement time')
#     ax_measurement_time.set_ylabel('Substrate label')
#     ax_measurement_time.set_title(experiment_name)
# =============================================================================

# =============================================================================
#     """Measurement time (or processing time)"""
#     fig_measurement_time = plt.figure(2)
#     fig_measurement_time.set_size_inches(12,8)
#     fig_measurement_time.set_tight_layout(True)
#     ax_measurement_time = plt.gca()  
#     sns.stripplot(y=big_data_df['Substrate label'], x=mpl.dates.date2num(big_data_df['Measurement time (datetime)']), ax=ax_measurement_time, color=violincolour)
#     ax_measurement_time.set_xlabel('Substrate label')
#     ax_measurement_time.set_ylabel('Measurement time (datetime)')
#     ax_measurement_time.set_title(experiment_name)
#     ax_measurement_time.xaxis.set_major_formatter(mpl.dates.DateFormatter('%d/%/%y %H:%M'))
# #    ax_measurement_time.xaxis.set_major_locator(mpl.dates.DayLocator(interval=10))
#     fig_measurement_time.autofmt_xdate()
# =============================================================================

